import json
import logging
import os.path

from infra.ya_salt.lib import init_manager
from infra.ya_salt.lib import persist
from infra.ya_salt.lib import package_manager
from infra.ya_salt.lib import pbutil
from infra.ya_salt.lib import yasm_client
from infra.ya_salt.lib import envutil

from infra.ya_salt.lib.components import apt
from infra.ya_salt.lib.components import kernel
from infra.ya_salt.lib.components import node_info
from infra.ya_salt.lib.components import server_info
from infra.ya_salt.lib.components import yasm
from infra.ya_salt.lib.reporters import monitoring
from infra.ya_salt.lib.components import manager

log = logging.getLogger('hostmanager')

KERNEL_PACKAGE_NAME = 'linux-image-server'
HOSTCTL_YASM_FLAG_PATH = '/run/.yasm-managed-by-hostctl'


def handle_init_failure(spec, status, err):
    logging.fatal('==== Fatal error: {} ===='.format(err))
    # Fail all salt states
    status.salt.Clear()
    pbutil.false_cond(status.salt.init_ok, err)
    pbutil.false_cond(status.salt.compilation_ok, err)
    pbutil.false_cond(status.salt.execution_ok, err)
    log.info('Persisting status...')
    err = persist.save_status(status)
    if err is not None:
        log.error('Failed to save status: {}'.format(err))
    m = monitoring.Monitoring(spec, yasm_client, package_manager.Dpkg())
    m.notify(status)


def spec_from_grains(spec, grains):
    # Update grains itself
    spec.salt.grains_json = json.dumps(grains)
    # Manage gencfg
    del spec.gencfg_groups[:]
    spec.gencfg_groups.extend(i for i in grains.get('gencfg', []) if i.isupper())
    # Get walle_project
    prj = grains.get('walle_project')
    if not prj:
        log.error('No walle_project in grains, will use "unknown"')
        prj = 'unknown'
    spec.walle_project = prj
    spec.dc = grains.get('walle_dc', 'unknown')
    spec.location = grains.get('location', 'unknown')
    # Get walle_tags
    del spec.walle_tags[:]
    spec.walle_tags.extend(grains.get('walle_tags') or [])
    non_prod, err = envutil.non_prod_stage(spec.walle_tags)
    if err is not None:
        log.error('Cannot determine environment (assuming production): {}'.format(err))
    if non_prod:
        spec.env_type = 'prestable'
    else:
        spec.env_type = 'production'


def fill_spec_from_repo(spec, repo):
    spec.hostname = repo.get_hostname()
    spec.salt.environment = repo.get_env()


def send_metrics(spec, status, pkgman):
    m = monitoring.Monitoring(spec, yasm_client, pkgman)
    m.notify(status)


class HostManager(object):
    def __init__(self, spec, orly, pkg_man):
        self.spec = spec
        self.orly = orly
        self.init = init_manager.get_runtime_init()
        self.pkg_man = pkg_man

    @staticmethod
    def run_server_info(ctx, salt_spec, status):
        if ctx.done():
            log.info('Skip server info component: {}'.format(ctx.error()))
        grains = json.loads(salt_spec.grains_json)
        s = server_info.ServerInfo(grains)
        s.run(status)

    def run_apt(self, ctx, apt_status):
        if ctx.done():
            log.info('Skip apt component: {}'.format(ctx.error()))
        a = apt.Apt(self.pkg_man)
        a.run(apt_status)

    def run_yasm(self, ctx, yasm_spec, yasm_status):
        if os.path.isfile(HOSTCTL_YASM_FLAG_PATH):
            log.info("Yasm is managed by hostctl, nothing to do...")
            return
        if not yasm_spec.agent_version:
            log.info('No yasm-agent version specified, nothing to do...')
            return
        if ctx.done():
            log.info('Skip yasm component: {}'.format(ctx.error()))
            return
        # Fow now we don't know how to check if we are 'production' or something else?
        # Maybe we can get it from server_info?
        y = yasm.Yasm(self.spec.env_type, self.pkg_man, self.orly)
        y.run(yasm_spec, yasm_status)

    def run_kernel(self, ctx, kspec_a, kspec_b, kernel_status):
        if ctx.done():
            log.info('Skip kernel component: {}'.format(ctx.error()))
            return
        k = kernel.Kernel(None)
        k.run(kspec_a, kspec_b, kernel_status)

    def run_node_info(self, ctx, ni_status):
        if ctx.done():
            return
        log.info('Running node info component...')
        ni = node_info.NodeInfo()
        ni.run(ni_status)

    def run(self, ctx, repo, spec_before, status, hctl):
        """
        :type repo: infra.ya_salt.lib.saltutil.SaltRepo
        :type ctx: infra.ya_salt.lib.context.ICtx

        :rtype: None|str
        :return: None if no errors where encountered or error string
        """
        if self.init.is_shutting_down():
            err = 'system is shutting down'
            log.info('Aborting execution: {}'.format(err))
            return err
        if ctx.done():
            log.info('Action cancelled: {}'.format(ctx.error()))
            return 'action cancelled: {}'.format(ctx.error())
        # node info - resources for cluster manager, etc.
        # Must be run **BEFORE** initial setup - because initial setup needs node info (boot id)
        self.run_node_info(ctx, status.node_info)
        log.info("Running: environment='{}'...".format(self.spec.env_type))
        log.info('Running apt component...')
        self.run_apt(ctx, status.apt)
        # server info - server_info.json
        log.info('Running server info component...')
        HostManager.run_server_info(ctx, self.spec.salt, status.server_info)
        # Run salt part after updating apt state,
        # but before kernel (to ensure salt installs proper kernel)
        # and before yasm (to ensure salt configures other yasm stuff (like config)
        log.info('Running salt...')
        mgr = manager.SaltManager(repo, self.spec, self.orly, status, hctl)
        mgr.run(ctx)
        log.info('Spec after salt execution:\n{}'.format(self.spec))
        log.info('Running yasm component...')
        self.run_yasm(ctx, self.spec.yasm, status.yasm)
        log.info('Running kernel component...')
        self.run_kernel(ctx, spec_before.kernel, self.spec.kernel, status.kernel)
        return None
