"""
Temporary (hopefully) layer to run hostmanager with different
salt implementation.
"""
import logging
import os


from infra.ya_salt.proto import ya_salt_pb2
from infra.ya_salt.lib import constants
from infra.ya_salt.lib import httpsalt
from infra.ya_salt.lib import info
from infra.ya_salt.lib import initial_setup
from infra.ya_salt.lib import orlyutil
from infra.ya_salt.lib import package_manager
from infra.ya_salt.lib import pbutil
from infra.ya_salt.lib import persist
from infra.ya_salt.lib import power_man
from infra.ya_salt.lib import reboots
from infra.ya_salt.lib import saltutil
from infra.ya_salt.lib import hostctl
from infra.ya_salt.lib import envutil

from . import hostmanager as hm

log = logging.getLogger('bootstrap')


class RunMode(object):
    """
    A way to get salt repository object.
    """

    def init_repo(self, grains):
        """
        :type grains: dict
        :rtype: saltutil.SaltRepo
        """
        raise NotImplementedError


class LocalMode(RunMode):
    def __init__(self, path):
        self._path = path

    def init_repo(self, grains):
        opts, err = saltutil.load_minion_config()
        if err is not None:
            return None, 'failed to load minion config: {}'.format(err)
        opts['grains'] = grains
        meta = ya_salt_pb2.LocalRepoMeta(commit_id='<local>', message="local dir='{}'".format(self._path))
        return saltutil.SaltRepo(meta=meta,
                                 config=opts,
                                 path=self._path), None


class FetchMode(RunMode):
    def __init__(self, repo_hosts):
        self.repo_hosts = repo_hosts

    def init_repo(self, grains):
        log.info('Initializing salt subsystem...')
        opts, err = saltutil.load_minion_config()
        if err is not None:
            return None, 'failed to load minion config: {}'.format(err)
        opts['grains'] = grains
        log.info('== Using http/local scheme for salt... ==')
        meta, err = httpsalt.HttpSalt(constants.LOCAL_REPO_BASEDIR,
                                      self.repo_hosts).sync()
        if err is not None:
            log.error("HTTP sync failed with '{}'".format(err))
        if meta is None:
            log.error('Local repo is not available - fail')
            return None, 'local repo is not available'
        # Inject overrides directory path if it exists.
        path = []
        if os.path.isdir(constants.REPO_OVERRIDE_DIR) and os.listdir(constants.REPO_OVERRIDE_DIR):
            path.append(constants.REPO_OVERRIDE_DIR)
        path.append(constants.LOCAL_REPO_CURRENT)
        return saltutil.SaltRepo(meta=meta, config=opts, path=path), None


def run_reboot_man(ctx, salt_config_pb, spec, status):
    if ctx.done():
        log.info('Skip reboot manager: {}'.format(ctx.error()))
        return
    log.info('Running reboot manager...')
    if salt_config_pb.no_orly is False:
        orly = orlyutil.Orly(spec.hostname,
                             labels={'ctype': spec.env_type,
                                     'prj': spec.walle_project,
                                     'geo': spec.location,
                                     },
                             url=salt_config_pb.orly_url)
    else:
        orly = None
    rm = reboots.RebootManager(orly, power_man.PowerManager())
    rm.run(spec, status)


def init_orly(spec, salt_config_pb, orly_cls=orlyutil.Orly):
    if salt_config_pb.no_orly is False:
        return orly_cls(spec.hostname,
                        labels={'ctype': spec.env_type,
                                'prj': spec.walle_project,
                                'geo': spec.location,
                                },
                        url=salt_config_pb.orly_url)
    return None


def do_run_hm(ctx, spec, status, repo, config_pb, pkgman, hctl):
    spec_before, err = persist.load_spec()
    if err is not None:
        log.warn('Failed to load spec: {}'.format(err))
        log.info('Will use clean spec')
        spec_before = ya_salt_pb2.HostmanSpec()
    # Fill in some spec fields from grains, to be able to report
    # to yasm with proper tags.
    # Salt adds more grains (gencfg in particular), re-run it
    hm.fill_spec_from_repo(spec, repo)
    orly = init_orly(spec, config_pb)
    h = hm.HostManager(spec, orly, pkgman)
    return h.run(ctx, repo, spec_before, status, hctl)


def run_hm(ctx, config_pb, mode=None):
    """
    Executes main script:
      * hostmanager (mainly salt part)
      * hostctl part
      * initial setup routines
      * reporting
      * rebooting
    """
    # ** load and init phase **
    hctl = hostctl.Hostctl(config_pb)
    status, err = persist.load_status()
    if err is not None:
        log.info('Failed to load status: {}'.format(err))
        log.info('Will use clean status')
        status = ya_salt_pb2.HostmanStatus()
    grains, err = info.load_grains()
    if err is not None:
        return None, "failed to gather grains: {}".format(err)
    spec = ya_salt_pb2.HostmanSpec()
    # We need to extract some information from grains, to
    # be able to report failure in yasm with proper tags.
    hm.spec_from_grains(spec, grains)
    # If report addresses empty use automatic discovery
    if not config_pb.report_addrs:
        ra, err = envutil.report_addr_from_tags_location(spec.walle_tags, spec.location)
        if err is not None:
            m = 'failed to get report addrs: {}'.format(err)
            hm.handle_init_failure(spec, status, m)
            return None, m
        if not ra:
            m = 'failed to get report addrs: empty result'
            hm.handle_init_failure(spec, status, m)
            return None, m
        config_pb.report_addrs.extend(ra)
    repo_hosts, err = envutil.repo_hosts_from_tags_location(spec.walle_tags, spec.location)
    if err is not None:
        m = 'failed to get salt masters: {}'.format(err)
        hm.handle_init_failure(spec, status, m)
        return None, m
    if mode is None:
        mode = FetchMode(repo_hosts)
    # Try to initialize salt
    repo, err = mode.init_repo(grains)
    if err is not None:
        m = 'failed to init salt: {}'.format(err)
        hm.handle_init_failure(spec, status, m)
        return None, m
    opts, err = repo.get_opts()
    if err:
        return None, 'failed to get repo opts: {}'.format(err)
    # After repo init we should rewrite spec with new grains
    hm.spec_from_grains(spec, opts['grains'])
    spec.salt.revision = repo.get_meta().commit_id
    pkgman = package_manager.Dpkg()
    pbutil.true_cond(status.salt.init_ok)
    # ===== <Note> =====
    # This is tricky part: order matters and it is hard to test
    # - thus it is fragile
    # Main theme: **order does matter**
    # ===== </Note> =====
    # ** Let main action begin - run salt and friends **
    err = do_run_hm(ctx, spec, status, repo, config_pb, pkgman, hctl)
    if err is not None:
        log.error("Hostmanager execution failed: {}".format(err))
    # *** salt/node-info part done ***
    # *** all active routines finished, let's postprocess results ***
    initial_setup.InitialSetup().run(spec, status)
    # *** now post process results by signaling external entities ***
    hm.send_metrics(spec, status, pkgman)
    # *** all routines changing spec/status are done - persist ***
    sp_err = persist.save_spec(spec)
    if sp_err is not None:
        log.error("Failed to save spec: {}".format(sp_err))
    st_err = persist.save_status(status)
    if st_err is not None:
        log.error("Failed to save spec: {}".format(sp_err))
    # **Attention**: run reboot manager after saving status
    run_reboot_man(ctx, config_pb, spec, status)
    log.info('==== Done ====')
    return status, err
