import logging
import os.path
import json

import salt.fileclient
from infra.ya_salt.lib import constants
from infra.ya_salt.lib import saltutil
from infra.ya_salt.lib.components import manager
from infra.ya_salt.lib.packages import stateutil
from infra.ya_salt.proto import ya_salt_pb2
from . import hostmanager
from . import shim

log = logging.getLogger(__name__)


# Example repo hierarchy
# /var/lib/ya-salt/exec/porto/porto-5-0-21/init.sls
# path = /var/lib/ya-salt/exec
# roots = ['porto']
# component_name = porto-5-0-21
class SaltExecRepo(saltutil.SaltRepo):
    def lo_from_selector(self, selector):
        return self.empty_lo(), None

    def persist_lo_for_selector(self, lo, selector):
        return None

    def purge_lo_for_selector(self, selector):
        return None

    @staticmethod
    def init_opts(opts, path, roots):
        opts['environment'] = roots[0]
        opts = saltutil.SaltRepo.init_opts(opts, path, roots)
        return opts

    def __init__(self, meta, config, path, roots=None, file_client_cls=salt.fileclient.LocalClient, inject_grains=False,
                 component_name=None):
        self._component_name = component_name
        super(SaltExecRepo, self).__init__(meta, config, path, roots, file_client_cls, inject_grains)

    def list_selectors(self):
        component_type = self._roots[0]
        selectors = [saltutil.Selector(component_type, self._component_name)]
        return selectors, None


class LocalExecMode(shim.RunMode):
    def __init__(self, path, component_type, component_name):
        self._path = path
        self._component_type = component_type
        self._component_name = component_name

    def init_repo(self, grains):
        opts, err = saltutil.load_minion_config()
        if err is not None:
            return None, 'failed to load minion config: {}'.format(err)
        opts['grains'] = grains
        meta = ya_salt_pb2.LocalRepoMeta(commit_id='<exec>', message="local dir='{}'".format(
            os.path.join(self._path, self._component_type, self._component_name)))
        return SaltExecRepo(meta=meta,
                            config=opts,
                            path=self._path, roots=[self._component_type], component_name=self._component_name), None


class SaltExecManager(manager.SaltManager):

    def __init__(self, repo, spec_b, orly, status, hctl):
        super(SaltExecManager, self).__init__(repo, spec_b, orly, status, hctl)

    def prepare_tx(self, components):
        for c in components:
            stateutil.add_packages_from_component(self.tx, c)


def load_grains():
    try:
        with open(constants.INFO_FILE_PATH) as f:
            return json.load(f), None
    except Exception:
        err = 'failed to load grains from {}'.format(constants.INFO_FILE_PATH)
        log.exception(err)
        return None, err


def run_exec(ctx, path):
    log.info('Running salt...')
    status = ya_salt_pb2.HostmanStatus()
    grains, err = load_grains()
    if err is not None:
        m = 'failed to init salt: {}'.format(err)
        return None, m
    parts = path.split('/')
    path = '/'.join(parts[:-2])
    component_type = parts[-2]
    component_name = parts[-1]
    spec = ya_salt_pb2.HostmanSpec()
    repo, err = LocalExecMode(path, component_type, component_name).init_repo(grains)
    if err is not None:
        m = 'failed to init salt: {}'.format(err)
        hostmanager.handle_init_failure(spec, status, m)
        return None, m
    hostmanager.fill_spec_from_repo(spec, repo)
    hostmanager.spec_from_grains(spec, grains)
    opts, err = repo.get_opts()
    if err:
        return None, 'failed to get repo opts: {}'.format(err)
    # After repo init we should rewrite spec with new grains
    hostmanager.spec_from_grains(spec, opts['grains'])
    if err is not None:
        return None, "failed to gather grains: {}".format(err)
    mgr = SaltExecManager(repo, spec, None, status, None)
    ok = mgr.run(ctx)
    if not ok:
        return None, err
    return status, None
