package main

import (
	"bytes"
	"encoding/json"
	"fmt"
	"os/exec"
	"time"
)

type JugglerEvent struct {
	Description string `json:"description"`
	Service     string `json:"service"`
	Status      string `json:"status"`
}

type JugglerEvents struct {
	Events []JugglerEvent `json:"events"`
}

type HostmanStatus struct {
	Salt struct {
		ExecutionOk struct {
			Message string `json:"message"`
			Status  string `json:"status"` // True|False|Unknown
		} `json:"executionOk"`
		Started string `json:"started"`
	} `json:"salt"`
}

func getHostmanStatus() HostmanStatus {
	var stdout, stderr bytes.Buffer
	status := HostmanStatus{}

	cmd := exec.Command("ya-salt", "status", "-o", "json")
	cmd.Stdout = &stdout
	cmd.Stderr = &stderr

	if err := cmd.Run(); err != nil {
		status.Salt.ExecutionOk.Status = "False"
		status.Salt.ExecutionOk.Message = err.Error()
		if stderr.Len() > 0 {
			status.Salt.ExecutionOk.Message += ": " + stderr.String()
		}
		return status
	}

	if err := json.Unmarshal(stdout.Bytes(), &status); err != nil {
		status.Salt.ExecutionOk.Status = "False"
		status.Salt.ExecutionOk.Message = "Can't parse status: " + err.Error()
	}

	return status
}

func main() {
	event := JugglerEvent{
		Description: "Ok",
		Service:     "hostman",
		Status:      "OK",
	}

	status := getHostmanStatus()

	if status.Salt.ExecutionOk.Status != "True" {
		event.Status = "CRIT"
		event.Description = status.Salt.ExecutionOk.Message
	} else {
		startTime, err := time.Parse(time.RFC3339, status.Salt.Started)
		if err != nil || time.Now().After(startTime.Add(time.Hour/2)) {
			event.Status = "CRIT"
			event.Description = "Runned more than half an hour ago"
		}
	}

	events := JugglerEvents{Events: []JugglerEvent{event}}
	jsonBytes, err := json.Marshal(events)
	if err != nil {
		panic(err.Error())
	}

	fmt.Println(string(jsonBytes))
}
