import errno
import logging
import os
import random
import time

from infra.ya_salt.lib import pbutil

log = logging.getLogger('apt-component')


class Apt(object):
    """
    Updates apt database by calling apt-get update.

    :type pack_man: infra.ya_salt.lib.package_manager.PackageManager
    """
    PERIOD_SECONDS = 20 * 60  # How often run update
    JITTER_SECONDS = 20 * 60  # Jitter added to update
    MAX_FORWARD_SECONDS = 24 * 60 * 60  # How long can we delay

    UPDATES_DIR = '/var/lib/dpkg/updates/'

    @classmethod
    def calc_next_update(cls, time_func=time.time):
        now = time_func()
        return int(now) + cls.PERIOD_SECONDS + random.randint(0, cls.JITTER_SECONDS)

    def __init__(self, pack_man):
        self.pack_man = pack_man

    def _run_update(self, last_update_ok):
        err = self.pack_man.update()
        if err is not None:
            log.error('Failed to sync package index: {}'.format(err))
            pbutil.set_condition(last_update_ok, 'False', err)
        else:
            pbutil.set_condition(last_update_ok, 'True')

    def _run_fix(self, listdir=os.listdir):
        """
        Performs check similar to apt-pkg/deb/debsystem.cc:CheckUpdates,
        i.e. tries to find files in /var/lib/dpkg/updates/ which are **all** numerical, e.g. "002"
        and fixes by running `dpkg --configure -a`.
        """
        try:
            dirents = listdir(self.UPDATES_DIR)
        except EnvironmentError as e:
            log.warning('failed: ls({}) = {}'.format(self.UPDATES_DIR, errno.errorcode.get(e.errno)))
            return str(e)
        damaged = False
        for dirent in dirents:
            if dirent.isdigit():
                damaged = True
                break
        if not damaged:
            return None
        err = self.pack_man.repair_half_configured()
        if err:
            return err
        return None

    def run(self, apt_status):
        now = time.time()
        next_ = apt_status.next_update_run_time.ToSeconds()
        # If host time was very off (e.g. booted in June, while it was May)
        # we can set next time very far off and fail to install new packages
        # for a month.
        if now < next_ and (next_ - now) < self.MAX_FORWARD_SECONDS:
            return
        self._run_update(apt_status.last_update_ok)
        self._run_fix()
        apt_status.next_update_run_time.FromSeconds(self.calc_next_update())
        apt_status.last_update_duration.FromSeconds(int(time.time() - now))
