"""
Kernel management component.
"""
import logging
import yaml

from infra.ya_salt.lib import kernel_man
from infra.ya_salt.lib import pbutil
from infra.ya_salt.lib import constants

log = logging.getLogger('kernel-component')


def update_boot_version(status, version_getter):
    v, err = version_getter()
    if err is not None:
        if status.version != 'unknown':
            status.transition_time.GetCurrentTime()
        status.version = 'unknown'
        status.error = err
    else:
        if status.version != v:
            status.transition_time.GetCurrentTime()
        status.version = v
        status.error = ''


def get_kernel_version_from_hostctl_spec(path=constants.HOSTCTL_KERNEL_SPEC_PATH, open_file=open):
    try:
        with open_file(path) as f:
            return yaml.safe_load(f)[constants.HOSTCTL_KERNEL_FIELD], None
    except OSError as e:
        return '', 'failed to open {}: {}'.format(path, e)
    except KeyError:
        return '', 'kernel spec has no "{}" field'.format(constants.HOSTCTL_KERNEL_FIELD)
    except Exception as e:
        return '', 'unknown error loading {}: {}'.format(path, e)


class Kernel(object):
    def __init__(self, k_manager):
        self.kernel_man = k_manager or kernel_man.Manager(None)

    def run(self, spec_a, spec_b, status):
        version = self.kernel_man.get_version()
        update_boot_version(status.boot_version, self.kernel_man.get_boot_version)
        spec_b.version, err = get_kernel_version_from_hostctl_spec()
        if err:
            log.error('Failed to get kernel version from hostctl spec: {}'.format(err))
        spec_version = spec_b.version or spec_a.version
        # TODO: remove this dirty hack
        # prevent flapping if spec_b has no version due to orly deny
        spec_b.version = spec_version
        if not spec_version:
            # Update version in our status
            status.version = version
            pbutil.set_condition(status.need_reboot, 'False', 'No version in spec')
            pbutil.set_condition(status.ready, 'Unknown', 'No version in spec')
            return
        if spec_version == version:
            # We might have just got from reboot, set reboot status
            if version != status.version:
                # Update version in our status
                status.version = version
            pbutil.set_condition(status.ready, 'True')
            pbutil.set_condition(status.need_reboot, 'False', 'Spec and current versions equal')
            return
        status.version = version
        pbutil.set_condition(status.ready, 'False',
                             "spec.version='{}' != status.version='{}'".format(spec_version, status.version))
        # Need to reboot, but can we?
        if status.boot_version.version != spec_version:
            message = "Invalid boot version: '{}' != '{}'".format(
                status.boot_version.version, spec_version,
            )
            pbutil.set_condition(status.need_reboot, 'False', message)
        else:
            pbutil.set_condition(status.need_reboot, 'True', 'Ready to reboot into {}'.format(spec_version))
