import logging
import time

from infra.ya_salt.lib import pbutil, saltutil, hostctl as lib_hostctl
from infra.ya_salt.lib.components import hostctl
from infra.ya_salt.lib.components import salt_component
from infra.ya_salt.lib.components import virtual
from infra.ya_salt.lib.packages import stateutil, transaction, transaction_validator
from infra.ya_salt.proto import ya_salt_pb2

log = logging.getLogger('salt-manager')


def is_components_noop(components):
    for c in components:
        # components are noop only if orly check failed
        if c.is_noop():
            return True
    return False


class ComponentCompiler(object):
    def __init__(self, repo, salt_status, orly, spec_b, hctl):
        self.repo = repo
        self.status = salt_status
        self.orly = orly
        self.spec_b = spec_b
        self._compiler = salt_component.Compiler(repo, salt_status, orly, spec_b, hctl)
        hctl_orly_url = lib_hostctl.orly_url_from_orly(orly,
                                                       self.status.initial_setup_passed) if orly and salt_status else None
        self._virtual_compiler = virtual.Compiler(self.status, hctl)
        self._hostctl_compiler = hostctl.Compiler(salt_status, hctl, hctl_orly_url)

    def compile(self, selector):
        type_ = selector.get_type()
        if type_ == 'salt':
            return self._compiler.compile(selector)
        elif type_ == 'virtual':
            return self._virtual_compiler.compile(selector)
        elif type_ == 'hostctl':
            return self._hostctl_compiler.compile(selector)
        else:
            return None, 'component type {} not supported, selector: {}'.format(type_, selector)


class SaltManager(object):
    def __init__(self, repo, spec_b, orly, status, hctl):
        self.repo = repo
        self.spec_b = spec_b
        self.status = status
        self.tx = transaction.Transaction()
        self.prev_status = ya_salt_pb2.HostmanStatus()
        self.prev_status.CopyFrom(self.status)
        self.compiler = ComponentCompiler(self.repo, status, orly, spec_b, hctl)
        # we should always clean up old statuses
        del self.status.salt_components[:]
        del self.status.salt.state_results[:]

    def get_selectors(self):
        return self.repo.list_selectors()

    def compile_components(self, selectors):
        components = []
        errors = []
        for s in selectors:
            c, err = self.compiler.compile(s)
            if err:
                errors.append(err)
            else:
                components.append(c)
        if errors:
            errors = '; '.join(errors)
        return components, errors

    def prepare_tx(self, components):
        for c in components:
            stateutil.add_packages_from_component(self.tx, c)

    def apply_components(self, components, ctx):
        results = []
        for c in components:
            if ctx.done():
                log.info('Stop components application: {}'.format(ctx.error()))
                break
            results.append(c.apply(ctx))
        return results

    @staticmethod
    def _salt_execution_succeeded(results):
        rv = True
        for r in results:
            rv &= r.applied()
        return rv

    def process_results(self, components):
        for c in components:
            if c.applied():
                c.process_results()

    def cleanup(self):
        prev_matches = {s.selector for s in self.prev_status.salt_components}
        curr_matches = {s.selector for s in self.status.salt_components}
        for match in (prev_matches - curr_matches):
            selector = saltutil.Selector(None, match)
            log.debug('Cleaning lo for: {}'.format(selector))
            err = self.repo.purge_lo_for_selector(selector)
            if err:
                log.error('Cannot purge lo for {}: {}'.format(selector, err))

    def run(self, ctx):
        salt_status = self.status.salt
        if self.repo.has_overrides():
            pbutil.true_cond(self.status.salt.overridden)
        else:
            pbutil.false_cond(self.status.salt.overridden, '')
        log.info('Search path: {}'.format(':'.join(self.repo.get_path())))
        salt_status.started.GetCurrentTime()
        selectors, err = self.get_selectors()
        if err:
            msg = 'Cannot get salt selectors: {}'.format(err)
            log.error(msg)
            pbutil.false_cond(salt_status.compilation_ok, msg)
            pbutil.false_cond(salt_status.execution_ok, msg)
            salt_status.last_run_duration.FromMilliseconds(
                int(time.time() * 1000) - salt_status.started.ToMilliseconds()
            )
            return False
        components, err = self.compile_components(selectors)
        if err:
            msg = 'Cannot initialize components: {}'.format(err)
            log.error(msg)
            pbutil.false_cond(salt_status.compilation_ok, msg)
        else:
            pbutil.true_cond(salt_status.compilation_ok)
        self.prepare_tx(components)
        errors = transaction_validator.validate_transaction(self.tx)
        if errors:
            msg = 'Cannot validate transaction: {}'.format('; '.join(errors))
            log.error(msg)
            pbutil.false_cond(salt_status.execution_ok, msg)
            salt_status.last_run_duration.FromMilliseconds(
                int(time.time() * 1000) - salt_status.started.ToMilliseconds()
            )
            return False
        results = self.apply_components(components, ctx)
        if not self._salt_execution_succeeded(results):
            log.error('Salt execution failed')
            pbutil.set_condition(salt_status.execution_ok, 'False')
            salt_status.last_run_duration.FromMilliseconds(
                int(time.time() * 1000) - salt_status.started.ToMilliseconds()
            )
            return False
        self.process_results(results)
        # Update global revision in status if there is no noop components
        if not is_components_noop(results):
            salt_status.revision = self.repo.get_meta().commit_id
            salt_status.revision_timestamp.CopyFrom(self.repo.get_meta().mtime)
        pbutil.true_cond(salt_status.execution_ok)
        salt_status.last_run_duration.FromMilliseconds(
            int(time.time() * 1000) - salt_status.started.ToMilliseconds()
        )
        log.info('Cleaning up missing lo states...')
        self.cleanup()
        return True
