"""
Component which generates /etc/server_info.json used by
  * oops??
  * iss??
  * juggler checks
  * yasm-agent
  extracts wall-e project.
"""
import json
import logging
import os
import random
import time

from infra.ya_salt.lib import constants
from infra.ya_salt.lib import fileutil
from infra.ya_salt.lib import lldputil
from infra.ya_salt.lib import pbutil
from infra.ya_salt.lib import points_info

log = logging.getLogger('server-info')


class Lui(object):
    DEFAULT_LUI_PATH = '/etc/lui.json'

    @classmethod
    def from_disk(cls, path=DEFAULT_LUI_PATH):
        try:
            with open(path) as f:
                d = json.load(f)
                ts = os.fstat(f.fileno()).st_mtime
            return cls(
                d['name'],
                int(ts),
            )
        except Exception as e:
            log.exception(e)
            return cls('unknown', 0)

    def __init__(self, name, ts_seconds):
        self.name = name
        self.ts_seconds = ts_seconds

    def to_info(self):
        return {
            'name': self.name,
            'timestamp': self.ts_seconds,
        }


class ServerInfo(object):
    PERIOD_SECONDS = 20 * 60  # How often run update
    JITTER_SECONDS = 10 * 60  # Jitter added to update
    EXCLUDE = frozenset(
        ['groupname',
         'path',
         'pid',
         'ps',
         'pythonexecutable',
         'pythonpath',
         'pythonversion',
         'saltpath',
         'shell',
         'uid',
         'username',
         'zmqversion',
         'master',
         'locale_info',
         'lldp',
         ]
    )

    @classmethod
    def calc_next_update(cls, time_func=time.time):
        now = time_func()
        return int(now) + cls.PERIOD_SECONDS + random.randint(0, cls.JITTER_SECONDS)

    def __init__(self, grains):
        self.grains = grains

    @staticmethod
    def _update_info(info, last_update_ok, info_path=constants.INFO_FILE_PATH):
        buf = json.dumps(info, indent=4, sort_keys=True)
        prev, err = fileutil.read_file(info_path, 100 * 1024)
        if err is not None:
            # Maybe it does not exists or something - try to rewrite it.
            log.info('Failed to read {}: {}'.format(info_path, err))
        elif prev == buf:
            pbutil.set_condition(last_update_ok, 'True')
            return
        log.info('Saving server info in {}...'.format(info_path))
        err = fileutil.atomic_write(info_path, buf)
        if err is not None:
            log.error('Failed to save: {}'.format(err))
            pbutil.set_condition(last_update_ok, 'False', err)
        else:
            pbutil.set_condition(last_update_ok, 'True')

    @staticmethod
    def _generate_info(grains, _recv_lldp=lldputil.recv_lldp):
        info = {'lui': Lui.from_disk().to_info()}
        for name, value in grains.items():
            if name in ServerInfo.EXCLUDE:
                continue
            # Special case - we store [16,4] - maj/minor in a list.
            if isinstance(value, list) and name != 'osrelease_info':
                value.sort()
            info[name] = value
        log.info("Receiving lldp info...")
        lldp, err = _recv_lldp()
        if err is not None:
            log.exception("Failed to receive lldp: {}".format(err))
            lldp = lldputil.LldpInfo('unknown', 'unknown')
        info['lldp'] = [
            {'switch': lldp.sysname, 'port': lldp.port},
        ]
        log.info("Gathering points info...")
        try:
            pi = points_info.get()
        except Exception:
            log.exception("Failed")
            pi = {}
        info['points_info'] = pi
        return info

    def run(self, status):
        now = time.time()
        # Update file if not does not exists or
        # it's been updated long ago
        if os.path.exists(constants.INFO_FILE_PATH):
            if now < status.next_update_run_time.ToSeconds():
                return
        info = self._generate_info(self.grains)
        self._update_info(info, status.last_update_ok)
        status.next_update_run_time.FromSeconds(self.calc_next_update())
