"""
YASM agent component.
"""
import logging
import time

import requests

from infra.ya_salt.lib import pbutil

log = logging.getLogger('yasm-component')


class Metrics(object):
    def __init__(self):
        self.agent_update_ok = 0
        self.agent_service_match = 0

    def to_yasm(self):
        return [
            # Current package install attempt ok/fail
            ('yasm-agent-install-ok_thhh', self.agent_update_ok),
            # Current agent reported version matches spec
            ('yasm-agent-service-match_thhh', self.agent_service_match),
        ]


class Yasm(object):
    AGENT_PACKAGE = 'yandex-yasmagent'
    HELPER_PACKAGE = 'yandex-search-yasmagent'
    DEFAULT_HELPER_VERSION = '1.0-2'
    ORLY_RULE_TEMPLATE = 'hostman-yasm-{}'
    AGENT_VERSION_URL = 'http://localhost:11003/version/'

    def __init__(self, env_name, pkg_man, orly):
        self.env_name = env_name or 'production'
        self.orly_rule = self.ORLY_RULE_TEMPLATE.format(self.env_name)
        self.pkg_man = pkg_man
        self.orly = orly
        self._m = Metrics()

    def run_helper_package(self, version, status):
        if not version:
            version = self.DEFAULT_HELPER_VERSION
        pkg, err = self.pkg_man.get_package_status(self.HELPER_PACKAGE)
        # Failed to check current version
        if err is not None:
            status.version = 'unknown'
            pbutil.set_condition(status.last_update_ok, 'False', err)
            return
        if pkg.version == version and pkg.installed:
            status.version = version
            pbutil.set_condition(status.last_update_ok, 'True')
            return
        status.version = pkg.version
        # Need to install new version
        pkg, err = self.pkg_man.install(self.HELPER_PACKAGE, version)
        if err is not None:
            pbutil.set_condition(status.last_update_ok, 'False', err)
            return
        status.version = pkg.version
        pbutil.set_condition(status.last_update_ok, 'True')

    def run_agent_service(self, status, http_get_func=requests.get):
        try:
            resp = http_get_func(self.AGENT_VERSION_URL, timeout=5)
        except Exception as e:
            pbutil.set_condition(status.last_check_ok, 'False', 'GET /version failed: {}'.format(e))
            return
        if not resp.ok:
            pbutil.set_condition(status.last_check_ok, 'False', 'Bad response code: {}'.format(resp.status_code))
            return
        # Strip response in case we have invalid output
        status.version = resp.content[:64]
        pbutil.set_condition(status.last_check_ok, 'True')

    def install_agent_package(self, version, status):
        """
        Installs agent package if needed and returns True if install was run succesfully.
        """
        pkg, err = self.pkg_man.get_package_status(self.AGENT_PACKAGE)
        # Failed to check current version
        if err is not None:
            status.version = 'unknown'
            pbutil.set_condition(status.last_update_ok, 'False', err)
            return False
        if pkg.version == version and pkg.installed:
            status.version = version
            pbutil.set_condition(status.last_update_ok, 'True')
            return False
        status.version = pkg.version
        if not version:
            pbutil.set_condition(status.last_update_ok, 'False', 'No version specified')
            return
        # Need to install new version
        if self.orly is not None:
            err = self.orly.start_operation(self.orly_rule)
            if err is not None:
                log.info('Need to install {}={} (got="{}"), but orly request failed: {}'.format(
                    self.AGENT_PACKAGE, version, pkg.version, err,
                ))
                pbutil.set_condition(status.last_update_ok, 'False', err)
                return False
            else:
                log.info('Orly allowed to install {}'.format(self.AGENT_PACKAGE))
        pkg, err = self.pkg_man.install(self.AGENT_PACKAGE, version)
        if err is not None:
            pbutil.set_condition(status.last_update_ok, 'False', err)
            return False
        self._m.agent_update_ok = 1
        status.version = pkg.version
        pbutil.set_condition(status.last_update_ok, 'True')
        return True

    def run(self, spec, status):
        # Set salt revision timestamp
        status.revision_timestamp = spec.revision_timestamp
        self.run_helper_package(spec.helper_version, status.helper_package)
        if self.install_agent_package(spec.agent_version, status.agent_package):
            # Give agent sometime to restart before checking agent status.
            time.sleep(5)
        self.run_agent_service(status.agent_service)
        if spec.agent_version == status.agent_service.version:
            self._m.agent_service_match = 1
        else:
            self._m.agent_service_match = 0

    def metrics(self):
        return self._m
