"""
ya-salt and hostctl environmental params self-identification routines

General idea is to extract stage value from wall-e tags and determine if stage belongs to production or prestable environment.
If stage belongs to production environment we are using production per-location settings.
If stage belongs to prestable environment we are using single prestable settings.

More reliable way to determine which master and report addresses should be used is to use wall-e tags similar to stage tags:
rtc.hm-sas, rtc.hm-prestable. On 2020-12-16 we could not use this method because we have enough cross-location wall-e projects
which could not be decomposed to location projects (rtc, rtc-testing, rtc-mtn, yp-xdc, rtc-mtn-hostman and so on...).

To update masters or report server lists change them in PROD_LOCATIONS_SETTINGS or PRESTABLE_SETTINGS dicts.

Details and discussion could be found at https://st.yandex-team.ru/HOSTMAN-779.
"""

REPO_HOSTS = 'repo_hosts'
REPORT_ADDRS = 'report'

PROD_LOCATIONS_SETTINGS = {
    'sas': {
        REPO_HOSTS: [
            'stout21.search.yandex.net',
            'sas1-4820.search.yandex.net',
            'sas1-4885.search.yandex.net',
            'sas1-5410.search.yandex.net',
            'sas2-9432.search.yandex.net',
        ],
        REPORT_ADDRS: [
            'stout21.search.yandex.net:8999',
            'sas1-4820.search.yandex.net:8999',
            'sas1-4885.search.yandex.net:8999',
            'sas1-5410.search.yandex.net:8999',
            'sas2-9432.search.yandex.net:8999',
        ],
    },
    'vla': {
        REPO_HOSTS: [
            'vla1-0474.search.yandex.net',
            'vla1-1451.search.yandex.net',
            'vla1-2999.search.yandex.net',
        ],
        REPORT_ADDRS: [
            'vla1-0474.search.yandex.net:8999',
            'vla1-1451.search.yandex.net:8999',
            'vla1-2999.search.yandex.net:8999',
        ],
    },
    'man': {
        REPO_HOSTS: [
            'man3-6180.search.yandex.net',
            'man0-0147.search.yandex.net',
            'man0-0136.search.yandex.net',
            'man0-0137.search.yandex.net',
            'man1-9007.search.yandex.net',
        ],
        REPORT_ADDRS: [
            'man0-0136.search.yandex.net:8999',
            'man3-6180.search.yandex.net:8999',
            'man0-0147.search.yandex.net:8999',
            'man0-0137.search.yandex.net:8999',
            'man1-9007.search.yandex.net:8999',
        ],
    },
    'msk': {
        REPO_HOSTS: [
            'myt0-3117.search.yandex.net',
            'myt0-0407.search.yandex.net',
            'myt0-3056.search.yandex.net',
        ],
        REPORT_ADDRS: [
            'myt0-3117.search.yandex.net:8999',
            'myt0-0407.search.yandex.net:8999',
            'myt0-3056.search.yandex.net:8999',
        ],
    },
}

PRESTABLE_SETTINGS = {
    REPO_HOSTS: [
        '42tzsve6rs5b6yzl.sas.yp-c.yandex.net',
        'gtfsouqjziwt5jwc.sas.yp-c.yandex.net',
        'utqg64qhk64inl45.vla.yp-c.yandex.net',
        'zhyk3dj6uzjnv77w.vla.yp-c.yandex.net',
    ],
    REPORT_ADDRS: [
        'hm-prestable-1.sas.yp-c.yandex.net:8999',
        'hm-prestable-2.vla.yp-c.yandex.net:8999',
        'hm-prestable-3.man.yp-c.yandex.net:8999',
    ],
}

PRESTABLE_STAGES = ('testing', 'prestable', 'experiment')
STAGE_TAG = 'rtc.stage-'


def stage_from_tags(walle_tags):
    """
    Extract stage value from wall-e tags: "rtc.stage-unknown" -> "unknown".
    :param walle_tags: list of wall-e tags
    :return: (stage, error)
    """
    env = None
    for t in walle_tags:
        if t.startswith(STAGE_TAG):
            if env:
                return None, 'multiple env walle_tags defined in tag set: {}'.format(walle_tags)
            else:
                env = t[len(STAGE_TAG):]
    return env, None


def non_prod_stage(walle_tags):
    """
    Determine if wall-e tags describe production host.
    :param walle_tags: list of wall-e tags
    :return: (bool, error). False if stage is prestable, True otherwise, None on error
    """
    stage, err = stage_from_tags(walle_tags)
    if err:
        return None, err
    if not stage:
        return None, 'cannot determine production or prestable environment from tag set: {}'.format(walle_tags)
    if stage in PRESTABLE_STAGES:
        return True, None
    else:
        return False, None


def _from_bundled_dict_tl(t, walle_tags, location):
    non_prod, err = non_prod_stage(walle_tags)
    if err is not None:
        return None, err
    if non_prod:
        return PRESTABLE_SETTINGS[t], None
    else:
        if location in PROD_LOCATIONS_SETTINGS:
            return PROD_LOCATIONS_SETTINGS[location][t], None
        else:
            return None, 'location "{}" not found in location settings'.format(location)


def repo_hosts_from_tags_location(walle_tags, location):
    """
    Get list of ya-salt repo hosts for combination of (walle_tags, location)
    :param walle_tags: list of wall-e tags
    :param location: wall-e/gencfg location (sas, vla, man, msk)
    :return: (masters, error)
    """
    return _from_bundled_dict_tl(REPO_HOSTS, walle_tags, location)


def report_addr_from_tags_location(walle_tags, location):
    """
    Get list of hm-report addresses for combination of (walle_tags, location)
    :param walle_tags: list of wall-e tags
    :param location: wall-e/gencfg location (sas, vla, man, msk)
    :return: (report_addresses, error)
    """
    return _from_bundled_dict_tl(REPORT_ADDRS, walle_tags, location)
