import errno
import os


def read_file(path, max_size=4 * 1024):
    """
    :rtype: (str|None, str|None)
    """
    try:
        with open(path) as f:
            if os.fstat(f.fileno()).st_size > max_size:
                return None, 'file is larger than max_size'
            return f.read(max_size), None
    except EnvironmentError as e:
        return None, str(e)


def write_all(buf, path):
    try:
        with open(path, 'w') as f:
            f.write(buf)
    except EnvironmentError as e:
        return str(e)


def atomic_write(path, buf, chmod=0644, make_dirs=False, times=None):
    if make_dirs:
        dir_name = os.path.dirname(path)
        if dir_name != '/':
            try:
                os.makedirs(dir_name)
            except EnvironmentError as e:
                if e.errno != errno.EEXIST:
                    return 'failed to create {}: {}'.format(dir_name, e)
    tmp = path + '.tmp'
    try:
        with open(tmp, 'w') as f:
            os.fchmod(f.fileno(), chmod)
            f.write(buf)
            f.flush()  # Flush to OS, before syncing
            os.fdatasync(f.fileno())
        if times is not None:
            os.utime(tmp, times)
        os.rename(tmp, path)
    except EnvironmentError as e:
        return "failed to write to '{}': {}".format(tmp, e)
    return None


def unlink(path):
    try:
        os.unlink(path)
    except OSError as e:
        return 'cannot unlink("{}"): {}'.format(path, e)
    return None
