import errno
import json
import logging
import os
import socket

import requests

from infra.ya_salt.lib import constants
from infra.ya_salt.lib import fileutil

TAG_KEY = 'gcfg_tag'
GENCFG_DATA_FILE = os.path.join(constants.VAR_LIB, 'gcfg.txt')
OLD_GENCFG_DATA_FILE = "/tmp/state/gcfg.txt"
# We return empty config so that states which check for gencfg groups
# will at least compile: e.g pillars (generic and for everyone).
# We can start failing, when we factor out common components away, i.e. get rid
# of pillars at least.
EMPTY_CONFIG = {
    'gcfg_tag': '',
    'gencfg': [],
}

log = logging.getLogger('gencfg-grains')


def _load_from_file(path):
    try:
        with open(path, 'r') as f:
            buf = f.read(4096)
    except EnvironmentError as e:
        return None, "failed to read from '{}': {}".format(path,
                                                           errno.errorcode.get(e.errno))
    try:
        return json.loads(buf), None
    except Exception as e:
        return None, "failed to parse JSON from '{}': {}".format(path, e)


def _transform_response(current_tag, d):
    """
    We must return a dictionary with keys:
        * gcfg_tag - put tag parameter in it
        * gencfg - put all eligible tags in a list
    """
    unique = set()
    el = d.get('instances_tags')
    if el is None:
        return None, 'no "instances_tags" in gencfg response'
    if not isinstance(el, dict):
        return None, '"instances_tags" is not a dict'
    for v in el.values():
        for tag in v:
            # Only add gencfg groups which are all upper cased.
            if tag.isupper():
                unique.add(tag)
    d = {
        TAG_KEY: current_tag,
        'gencfg': sorted(unique)
    }
    return d, None


def _load_from_api(tag, hostname, get_func=requests.get):
    gencfg_instance_url = "https://api.gencfg.yandex-team.ru/{}/hosts/{}/instances_tags"
    url = gencfg_instance_url.format(tag, hostname)
    try:
        response = get_func(url)
    except Exception as e:
        return None, 'request to {} failed: {}'.format(url, e)
    if response.status_code != 200:
        return None, 'request to {} failed with code {}'.format(url, response.status_code)
    try:
        d = json.loads(response.content)
    except Exception as e:
        return None, 'failed to parse JSON from gencfg response: {}'.format(e)
    return _transform_response(tag, d)


def _save_file(d, path):
    buf = json.dumps(d)
    return fileutil.atomic_write(path, buf, make_dirs=True)


def gencfg(tag, load_func=_load_from_api, load_from_cache=_load_from_file):
    """"
    Acquire/load gencfg state.
    """
    cfg, err = load_from_cache(GENCFG_DATA_FILE)
    if err is not None:
        # Try loading from old path and save to new.
        # BEWARE: But we do not remove old file in case we did to rollback!
        cfg, err = load_from_cache(OLD_GENCFG_DATA_FILE)
        if err is None:
            save_err = _save_file(cfg, GENCFG_DATA_FILE)
            if err is not None:
                log.error('Failed to save {}: {}'.format(GENCFG_DATA_FILE, save_err))
    if err is not None or TAG_KEY not in cfg or cfg[TAG_KEY] != tag:
        log.debug('Requesting {} from gencfg API...'.format(tag))
        cfg, err = load_func(tag, socket.gethostname())
        if err is not None:
            log.error('Failed to load tag from API: {}'.format(err))
            return EMPTY_CONFIG
        err = _save_file(cfg, GENCFG_DATA_FILE)
        if err is not None:
            log.error('Failed to save {}: {}'.format(GENCFG_DATA_FILE, err))
    cfg.pop(TAG_KEY)  # Do not leak gencfg tag to grains
    return cfg
