"""
Simple and ugly /boot/grub/grub.cfg parser.
"""
import re

MENU_ENTRY_RE = re.compile(r"^menuentry\s+'(.*?)'.*$")
SUBMENU_ENTRY_RE = re.compile(r'^submenu\s+["\'](.*?)["\'].*{\s?$')
LINUX_RE = re.compile(r'^linux\s+(?:/boot)?/vmlinuz-(\S+)\s.*$')


class MenuEntry(object):
    UNKNOWN = 0
    LINUX = 1

    def __init__(self, name):
        self.name = name
        self.os_type = self.UNKNOWN
        self.kernel_version = 'unknown'


class Grub1(object):
    def __init__(self):
        self.default_name = None
        self.menu_entries = []

    def get_boot_version(self):
        if not self.default_name:
            return None, 'failed to parse default name'
        if not self.menu_entries:
            return None, 'parsed no menu entries'
        if self.default_name.isdigit():
            idx = int(self.default_name)
            if idx > len(self.menu_entries) - 1:
                return None, "failed to eval default as index: {}".format(idx)
            return self.menu_entries[idx].kernel_version, None
        # Match by menu name
        for entry in self.menu_entries:
            if self.default_name == entry.name:
                return entry.kernel_version, None
        return None, "failed to find kernel version for '{}'".format(self.default_name)


MOD_GLOBAL = 0
MOD_MENU = 1
MOD_SUBMENU = 2


def parse_file(f):
    grub = Grub1()
    mode = MOD_GLOBAL
    menu_prefix = ''
    menu_entry = None
    for line in f:
        line = line.strip()
        if line.startswith('#'):
            continue
        if mode == MOD_GLOBAL:
            # We're looking for special env variable
            # https://www.gnu.org/software/grub/manual/grub/grub.html#default
            if line.startswith('set default='):
                prefix_len = len('set default=')
                if line[prefix_len] == '"':
                    # Find closing `"`
                    idx = line.find('"', prefix_len + 1)
                    if idx == -1:
                        return None, "failed to parse {}: no closing \"".format(line)
                    grub.default_name = line[prefix_len + 1:idx]
                else:
                    return None, "unsupported default format in '{}'".format(line)
            # https://www.gnu.org/software/grub/manual/grub/grub.html#submenu
            elif line.startswith('submenu '):
                m = SUBMENU_ENTRY_RE.match(line)
                if m is None:
                    return None, "failed to match '{}' as submenu".format(line)
                menu_prefix = m.group(1) + ">"
                mode = MOD_SUBMENU
            # https://www.gnu.org/software/grub/manual/grub/grub.html#menuentry
            elif line.startswith('menuentry '):
                m = MENU_ENTRY_RE.match(line)
                if m is None:
                    return None, "failed to match '{}' as menuentry".format(line)
                name = m.group(1)
                mode = MOD_MENU
                menu_entry = MenuEntry(menu_prefix + name)
        elif mode == MOD_MENU:
            if line.strip().startswith('}'):
                grub.menu_entries.append(menu_entry)
                menu_entry = None
                mode = MOD_GLOBAL
                continue
            m = LINUX_RE.match(line)
            if m is None:
                continue
            else:
                menu_entry.os_type = MenuEntry.LINUX
                menu_entry.kernel_version = m.group(1)
        elif mode == MOD_SUBMENU:
            if line.strip().startswith('}'):
                menu_prefix = ''
                mode = MOD_GLOBAL
                continue
            # https://www.gnu.org/software/grub/manual/grub/grub.html#menuentry
            elif line.startswith('menuentry'):
                # Here we assume title is always enclosed in `'`
                name = MENU_ENTRY_RE.match(line).group(1)
                if name is None:
                    return None, "failed to match '{}' as menuentry".format(line)
                mode = MOD_MENU
                menu_entry = MenuEntry(menu_prefix + name)
    return grub, None


def parse_grub_conf(path='/boot/grub/grub.cfg'):
    try:
        with open(path) as f:
            return parse_file(f)
    except EnvironmentError as e:
        return None, 'failed to read {}: {}'.format(path, e)
