import logging
import os
import time

from infra.ya_salt.lib import subprocutil
from infra.ya_salt.lib import constants
from infra.ya_salt.lib import pbutil

log = logging.getLogger('hostctl')
HOSTCTL_TIMEOUT = 30 * 60


class Hostctl(object):
    def __init__(self, config_pb):
        self._config = config_pb

    def available(self):
        if not self._config.hostctl_path:
            return 'hostctl is not available'
        return None

    def manage(self, repo_path, orly_url, check_output=subprocutil.check_output):
        err = self.available()
        if err:
            return err
        argv = [self._config.hostctl_path, 'manage',
                '--logfile', constants.LOG_FILE_HOSTCTL,
                '--repo', repo_path,
                '--report-addr', ','.join(self._config.report_addrs),
                ]
        if orly_url:
            argv.extend(['--orly-url', orly_url])
        else:
            argv.append('--no-orly')

        log.info('Executing hostctl manage: "{}"...'.format(' '.join(argv)))
        o, e, res = check_output(argv, timeout=HOSTCTL_TIMEOUT)
        if not res.ok:
            log.error('Hostctl manage execution failed: {}'.format(res.message))
            log.error('Stdout: {}'.format(o))
            log.error('Stderr: {}'.format(e))
            return res.message
        return None

    def report(self, check_output=subprocutil.check_output):
        err = self.available()
        if err:
            return err
        argv = [self._config.hostctl_path, 'report',
                '--logfile', constants.LOG_FILE_HOSTCTL,
                '--report-addr', ','.join(self._config.report_addrs),
                ]
        log.info('Executing hostctl report: "{}"...'.format(' '.join(argv)))
        o, e, res = check_output(argv, timeout=HOSTCTL_TIMEOUT)
        if not res.ok:
            log.error('Hostctl report execution failed: {}'.format(res.message))
            log.error('Stdout: {}'.format(o))
            log.error('Stderr: {}'.format(e))
            return res.message
        return None

    def manage_target(self, repo_path, target, orly_url, check_output=subprocutil.check_output):
        err = self.available()
        if err:
            return err
        argv = [
            self._config.hostctl_path, 'manage',
            '--logfile', constants.LOG_FILE_HOSTCTL,
            '--repo', repo_path,
            '--target-units', target,
        ]
        if orly_url:
            argv.extend(['--orly-url', orly_url])
        else:
            argv.append('--no-orly')
        log.info('Executing hostctl manage target for "{}": "{}"...'.format(target, ' '.join(argv)))
        o, e, res = check_output(argv, timeout=HOSTCTL_TIMEOUT)
        if not res.ok:
            log.error('Hostctl manage target execution failed: {}'.format(res.message))
            log.error('Stdout: {}'.format(o))
            log.error('Stderr: {}'.format(e))
            return res.message
        return None

    def manage_inline(self, name, repo_path, contents, check_output=subprocutil.check_output):
        err = self.available()
        if err:
            return err
        argv = [self._config.hostctl_path, 'manage',
                '--logfile', constants.LOG_FILE_HOSTCTL,
                '--repo', repo_path,
                '--no-orly',
                '--stdin',
                ]
        log.info('Executing hostctl manage for unit {}: "{}"...'.format(name, ' '.join(argv)))
        o, e, res = check_output(argv, timeout=HOSTCTL_TIMEOUT, data=contents)
        if not res.ok:
            log.error('Hostctl manage unit {} execution failed: {}'.format(name, res.message))
            log.error('Stdout: {}'.format(o))
            log.error('Stderr: {}'.format(e))
            return res.message
        return None


def orly_url_from_orly(orly, initial_setup_passed):
    # trim 'rest/' because hostctl append it by himself
    if orly and initial_setup_passed.status == 'True':
        return orly.orly.url if not orly.orly.url.endswith('rest/') else orly.orly.url[:-5]
    else:
        return None


def run_hostctl_manage(hctl, ctx, status, orly_url, repo_path=os.path.join(constants.VAR_LIB, '/repo/current')):
    """
    Run hostctl manage and save result to status
    :param hctl: Hostctl object
    :param ctx: run context
    :param status: HostmanStatus
    :param orly_url: str or None
    :param repo_path: str or None
    :return: err or None
    """
    if ctx.done():
        log.info('Skip hostctl: {}'.format(ctx.error()))
        return None
    err = hctl.available()
    if err:
        log.info('Not running hostctl manage: {}'.format(err))
        return err
    start = time.time()
    err = hctl.manage(repo_path, orly_url)
    status.hostctl.execution_time.FromMilliseconds(
        int((time.time() - start) * 1000)
    )
    if err:
        pbutil.false_cond(status.hostctl.ok, err)
        ctx.fail('virtual.hostctl', err)
        return err
    else:
        pbutil.true_cond(status.hostctl.ok, 'OK')
        ctx.ok('virtual.hostctl')
        log.info('Hostctl finished successfully')
    return None
