import calendar
import errno
import io
import itertools
import logging
import hashlib
import httplib
import random
import os
import shutil
import time
import zipfile

import requests
from google.protobuf import json_format

from infra.ya_salt.proto import ya_salt_pb2
from infra.ya_salt.lib import fileutil

log = logging.getLogger('httpsalt')


class Response(object):
    """
    Model specific server response. Transformed from http one.
    """
    def __init__(self, has_changed, content='', mtime=None, commit_id=None):
        self.has_changed = has_changed
        self.content = content
        self.mtime = mtime
        self.commit_id = commit_id

    @staticmethod
    def from_http(resp):
        """
        Validates http response and constructs new Response object.
        """
        modified = resp.headers.get('Last-Modified')
        commit_id = resp.headers.get('X-Commit-Id')
        if not modified:
            return None, 'no "Last-Modified" header in response'
        try:
            mtime = modified_to_seconds(modified)
        except Exception as e:
            return None, 'failed to parse "{}" as date: {}'.format(modified, e)
        try:
            body = resp.content
        except Exception as e:
            return None, 'failed to read response body: {}'.format(e)
        # Verify that we either got Content-Length or Chunked Encoding
        ce = resp.headers.get('Content-Encoding')
        if ce and 'chunked' in ce:
            # Chunks correctness check is done while consuming body
            return Response(True, body, mtime=mtime, commit_id=commit_id), None
        # Requests library does not check content length.
        # Here we **ASSUME** that content has not been encoded in any way
        # Because actually this is .zip file, no further encoding needed.
        cl = resp.headers.get('Content-Length')
        if not cl:
            return None, 'no content-length in response'
        try:
            l = int(cl)
        except Exception as e:
            return None, 'failed to parse content-length: {} as int: {}'.format(cl, e)
        if l != len(body):
            return None, 'incorrect response size: got={} expected={}'.format(len(body), l)
        return Response(True, body, mtime=mtime, commit_id=commit_id), None


def seconds_to_modified(seconds):
    d = time.gmtime(seconds)
    return '%s, %02d%s%s%s%s %02d:%02d:%02d GMT' % (
        ('Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat', 'Sun')[d.tm_wday],
        d.tm_mday, ' ',
        ('Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul', 'Aug', 'Sep',
         'Oct', 'Nov', 'Dec')[d.tm_mon - 1],
        ' ', str(d.tm_year), d.tm_hour, d.tm_min, d.tm_sec
    )


def modified_to_seconds(modified):
    # XXX: Our server always returns GMT time zone
    t = time.strptime(modified, "%a, %d %b %Y %H:%M:%S %Z")
    return calendar.timegm(t)


def gen_cookie(masters):
    h = hashlib.sha1()
    for s in sorted(masters):
        h.update(s)
    return h.hexdigest()


class HttpSalt(object):
    SALT_ZIP_URL = 'v1/salt/zip'
    DEFAULT_MASTER_PORT = 8080
    DEFAULT_MASTER_TIMEOUT = 30
    DEFAULT_ATTEMPTS_COUNT = 5
    SYNC_TIMEOUT_BASE = 6 * 3600

    @classmethod
    def get_sync_timeout(cls):
        """
        Timeout after which we request full zip in case files were corrupted.
        """
        return cls.SYNC_TIMEOUT_BASE + random.randint(0, 3600)

    @staticmethod
    def ensure_dir(output_dir):
        try:
            os.makedirs(output_dir)
        except EnvironmentError as e:
            if e.errno == errno.EEXIST:
                # It can be directory or file, we need to check
                if os.path.isdir(output_dir):
                    return None
                return "path '{}' exists and not a directory".format(output_dir)
            else:
                return "failed to create '{}': {}".format(output_dir, e)
        os.chmod(output_dir, 0755)
        return None

    @staticmethod
    def remove_all(output_dir):
        try:
            shutil.rmtree(output_dir, ignore_errors=False)
        except EnvironmentError as e:
            if e.errno != errno.ENOENT:
                return None
            return str(e)
        return None

    @staticmethod
    def fetch_content(url, headers, timeout=DEFAULT_MASTER_TIMEOUT, http_get_fun=requests.get):
        try:
            resp = http_get_fun(url, headers=headers, timeout=timeout)
        except Exception as e:
            return None, "GET {} failed: {}".format(url, e)
        if resp.status_code == httplib.NOT_MODIFIED:
            return Response(False), None
        if resp.status_code == httplib.OK:
            return Response.from_http(resp)
        return None, 'bad http code: {} {}'.format(resp.status_code, resp.reason)

    @staticmethod
    def save_meta(path, data):
        buf = json_format.MessageToJson(data)
        return fileutil.write_all(buf, os.path.join(path, 'meta.json'))

    @staticmethod
    def get_current(base_dir):
        """
        Returns current repo information or an error.
        """
        p = os.path.join(base_dir, 'current')
        try:
            dir_name = os.readlink(p)
        except EnvironmentError as e:
            if e.errno == errno.ENOENT:
                err = 'no current directory'
            else:
                err = 'failed to readlink({}): {}'.format(p, e)
            return None, err
        # We have current symlink pointing to existing directory
        # Check if it has meta.json
        meta_path = os.path.join(base_dir, dir_name, 'meta.json')
        buf, err = fileutil.read_file(meta_path)
        if err is not None:
            return None, err
        # Parse meta.json
        meta = ya_salt_pb2.LocalRepoMeta()
        try:
            json_format.Parse(buf, meta)
        except Exception as e:
            return None, 'failed to parse meta.json: {}'.format(e)
        return meta, None

    def __init__(self, output_dir, masters):
        self.output_dir = output_dir
        self.masters = masters

    def fetch_salt(self, mtime):
        random.shuffle(self.masters)
        headers = {
            b'User-Agent': b'ya-salt',
        }
        if mtime:
            headers[b'If-Modified-Since'] = seconds_to_modified(mtime)
        for i, master in enumerate(itertools.cycle(self.masters)):
            url = 'http://{}:{}/{}'.format(master, self.DEFAULT_MASTER_PORT, self.SALT_ZIP_URL)
            resp, err = self.fetch_content(url, headers)
            if err is None:
                return resp, None
            log.error('Failed to fetch salt.zip: {}'.format(err))
            if i < self.DEFAULT_ATTEMPTS_COUNT:
                time.sleep(i * 2)
            else:
                return None, err

    def unpack_files(self, content, meta):
        mtime = str(meta.mtime.seconds)
        path = os.path.join(self.output_dir, mtime)
        buf = io.BytesIO(content)
        # Remove path if it exists (partial downloads, re-syncs)
        try:
            shutil.rmtree(path)
        except EnvironmentError as e:
            if e.errno != errno.ENOENT:
                return 'failed to remove {}: {}'.format(path, e)
        except Exception as e:
            return 'failed to remove {}: {}'.format(path, e)
        try:
            zip_f = zipfile.ZipFile(buf)
            zip_f.extractall(path=path)
        except Exception as e:
            shutil.rmtree(path, ignore_errors=True)
            return 'failed to extract received zip: {}'.format(e)
        return self.save_meta(path, meta)

    def switch_symlink(self, mtime):
        mtime = str(mtime)
        path = os.path.join(self.output_dir, mtime)
        current = os.path.join(self.output_dir, 'current')
        # Create symlink with temporary name to atomically replace it
        cur_tmp_path = current + '.tmp'
        # But remove previous one if it has been left
        try:
            os.unlink(cur_tmp_path)
        except EnvironmentError as e:
            if e.errno != errno.ENOENT:
                return 'failed to remove {}: {}'.format(cur_tmp_path, e)
        try:
            os.symlink(mtime, cur_tmp_path)
        except EnvironmentError as e:
            shutil.rmtree(path, ignore_errors=True)
            return 'ln -s {} {}: {}'.format(path, cur_tmp_path, e)
        # Atomically replace symlink
        try:
            os.rename(cur_tmp_path, current)
        except EnvironmentError as e:
            shutil.rmtree(path, ignore_errors=True)
            return 'failed to symlink {} => current: {}'.format(mtime, e)
        # Garbage collect remnants of previous attempts
        for entry in os.listdir(self.output_dir):
            if entry in ['current', mtime]:
                continue
            rm_dir = os.path.join(self.output_dir, entry)
            try:
                shutil.rmtree(rm_dir)
            except EnvironmentError as e:
                log.error("Failed to remove '{}': {}".format(rm_dir, e))
        return None

    def sync(self):
        """
        Synchronize pillars and states from masters.

        Result is twofold:
          * can we use local repo copy?
          E.g. too old, no copy at all, failed during unpacking, making symlinks
          * have we encountered any error?
          E.g. old state is usable, but we failed to fetch/check new
        """
        err = HttpSalt.ensure_dir(self.output_dir)
        if err is not None:
            return None, 'sync failed: {}'.format(err)
        cookie = gen_cookie(self.masters)
        # Get current status of local repo
        meta, err = HttpSalt.get_current(self.output_dir)
        if err is not None:
            log.warning('Local repo is absent/corrupted: {}'.format(err))
            log.info('Will try to sync...')
            mtime = 0
        elif meta.server_cookie != cookie:
            log.warning('Server cookie changed, will drop local repo')
            err = self.remove_all(self.output_dir)
            if err is not None:
                return None, 'failed to invalidate repo: {}'.format(err)
            mtime = 0
            meta = None
        elif time.time() > meta.last_sync_ts.seconds + self.get_sync_timeout():
            log.info('Last sync was too long ago, will request full zip...')
            mtime = 0
        else:
            mtime = meta.mtime.seconds
            log.info('Current timestamp: {}'.format(mtime))
        resp, err = self.fetch_salt(mtime)
        if err is not None:
            return meta, 'sync failed: {}'.format(err)
        if not resp.has_changed:
            # Nothing to do
            log.info('Remote has not changed')
            return meta, None
        n_meta = ya_salt_pb2.LocalRepoMeta()
        n_meta.mtime.FromSeconds(int(resp.mtime))
        n_meta.commit_id = resp.commit_id or ''
        n_meta.last_sync_ts.GetCurrentTime()
        n_meta.server_cookie = cookie
        err = self.unpack_files(resp.content, n_meta)
        if err is not None:
            log.error('Failed to unpack: {}'.format(err))
            # Use old repo version
            return meta, err
        err = self.switch_symlink(resp.mtime)
        if err is not None:
            # If we failed to switch symlink, repo can be in
            # unusable state.
            return None, err
        return n_meta, None
