"""
Utilities to interact with init systems.
"""
import os

from infra.ya_salt.lib import subprocutil
from infra.ya_salt.lib import pbutil
from infra.ya_salt.proto import ya_salt_pb2


def _get_env():
    """
    Ensure locale because we parse initctl/systemctl output.
    """
    env = os.environ.copy()
    env['LANG'] = 'en_US.UTF-8'
    env['HOSTMAN'] = '1'
    return env


class Init(object):
    def name(self):
        raise NotImplementedError

    def get_running(self, name):
        """
        Queries service running status.

        :param unicode name: service name
        :rtype infra.ya_salt.proto.ya_salt_pb2.Condition:
        """
        raise NotImplementedError

    def restart(self, name):
        """
        Restarts service.

        :param unicode name: service name
        :rtype unicode|None:
        """
        raise NotImplementedError

    def stop(self, name):
        """
        Stops service.
        """
        raise NotImplementedError

    def start(self, name):
        """
        Starts service.
        """
        raise NotImplementedError

    def reload_config(self):
        """
        Asks init to reload its configuration.
        """
        raise NotImplementedError

    def is_shutting_down(self):
        """
        Checks if host is operating, i.e not shutting down.
        """
        raise NotImplementedError


class Systemd(Init):
    SYSTEMCTL = '/bin/systemctl'
    DEAD_STATES = ['dead', 'failed', 'exited']

    @classmethod
    def sub_to_cond(cls, sub, cond):
        """
        Updates condition according to systemctl SubState value.
        """
        if sub == 'running':
            pbutil.set_condition(cond, "True", "running")
        elif sub in cls.DEAD_STATES:
            pbutil.set_condition(cond, "False", sub)
        else:
            pbutil.set_condition(cond, "Unknown", sub)

    def __init__(self, check_output=subprocutil.check_output):
        self.check_output = check_output

    def name(self):
        return 'systemd'

    def _get_sub_state(self, name):
        cmd = [
            self.SYSTEMCTL,
            'show',
            name + '.service',
            '-p',
            'SubState',
        ]
        stdout, stderr, status = self.check_output(cmd,
                                                   timeout=30)
        if not status.ok:
            return None, status.message
        buf = stdout.rstrip()
        if not buf.startswith('SubState='):
            return None, 'Unexpected output: {}'.format(buf[:30])
        return buf[len('SubState='):], None

    def get_running(self, name):
        c = ya_salt_pb2.Condition()
        sub, err = self._get_sub_state(name)
        if err is not None:
            pbutil.set_condition(c, "Unknown", err)
        else:
            self.sub_to_cond(sub, c)
        # Drop last transition, because we're not gathering it yet.
        c.transition_time.Clear()
        return c

    def start(self, name):
        # Just call start
        cmd = [
            self.SYSTEMCTL,
            'start',
            name + '.service',
        ]
        status = self.check_output(cmd, timeout=30)[2]
        if not status.ok:
            return status.message
        return None

    def stop(self, name):
        cmd = [
            self.SYSTEMCTL,
            'stop',
            name + '.service',
        ]
        status = self.check_output(cmd, timeout=30)[2]
        if not status.ok:
            return status.message
        return None

    def restart(self, name):
        cmd = [
            self.SYSTEMCTL,
            'restart',
            name + '.service',
        ]
        status = self.check_output(cmd, timeout=30)[2]
        if not status.ok:
            return status.message
        return None

    def reload_config(self):
        cmd = [
            self.SYSTEMCTL,
            'daemon-reload',
        ]
        status = self.check_output(cmd, timeout=30)[2]
        if not status.ok:
            return status.message
        return None

    def is_shutting_down(self):
        # See man 1 systemctl for documentation.
        # Host can be degraded (but this is not what we are looking for)
        # or it can be in `unknown` state (if we are deep in shutdown process)
        # or it can be in `stopping` state.
        # All states but `running` result it exit code 1, so we ignore it
        cmd = [
            self.SYSTEMCTL,
            'is-system-running',
        ]
        stdout, _, _ = self.check_output(cmd, timeout=30)
        return stdout.rstrip() in ['stopping', 'unknown']


def get_runtime_init():
    with open('/proc/1/comm') as f:
        exe = f.read()
    if exe == 'systemd\n':
        return Systemd()
    raise Exception("Unknown init in the system")
