"""
Statuses (in chronological order):
    * Not passed
    * Passed, need to record boot_id
    * Boot id recorded, waiting reboot
    * Rebooted, but needs to mark
    * Marked - finished
States can transition only from first to last, we cannot jump backwards.
As we do not have (for compatibility reasons) field to record current state, we can (and must) infer it from spec/status.
We do than backwards, i.e. check if we finished, then if we rebooted and need to mark and so on.
"""
import errno
import logging
import os
import time

from infra.ya_salt.lib import constants
from infra.ya_salt.lib import pbutil
from . import statusutil

log = logging.getLogger('initial-setup')


class Status(object):
    NOT_READY = 'Initial setup not passed, not ready for reboot'
    NEEDS_REBOOT = 'Needs reboot'
    REBOOT_DONE = 'Reboot done'
    PASSED = 'Passed'


class Solver(object):
    BROKEN_GRACE_PERIOD = 2 * 3600  # Do not reboot after so long

    @classmethod
    def is_not_ready_too_long(cls, status, time_func=time.time):
        """
        Checks that we do not reboot on hosts which where broken for long enough
        and all services are running already.
        """
        # empty status means first run
        if not status.initial_setup_passed.status:
            return False

        p = time_func() - status.initial_setup_passed.transition_time.ToSeconds()
        return p > cls.BROKEN_GRACE_PERIOD

    @classmethod
    def kernel_is_ready_or_need_reboot(cls, k):
        if k.ready.status == 'True':
            return True
        if k.need_reboot.status != 'True':
            return False
        return True

    @classmethod
    def ready_err(cls, spec, status):
        # After host redeployment/initial install we have base environment
        # which after first run switches to "search_runtime".
        if status.initial_setup_passed.status == 'True':
            # Assume finished
            return None
        if not spec.salt.environment:
            return 'empty environment in salt'
        if status.salt.execution_ok.status != 'True':
            # If we need to perform initial setup and salt execution
            # failed - don't drop flag, need to try again
            return 'salt execution failed'
        if statusutil.get_executed_states_count(status.salt_components) == 0:
            return 'no salt states executed'
        if statusutil.has_failed_executed_states(status.salt_components):
            return 'some salt states are failed'
        if cls.is_not_ready_too_long(status):
            return 'failed to pass initial setup for too long'
        # Assume finished (i.e. need to proceed to reboot)
        # if kernel is not ready but needs reboot
        # (i.e. proper kernel version is installed)
        if not cls.kernel_is_ready_or_need_reboot(status.kernel):
            return 'kernel not ready and not ready for reboot'
        if status.hostctl.ok.status != 'True':
            return 'hostctl execution failed: {}'.format(status.hostctl.ok.message)
        return None


def fsm_passed(_, spec, status):
    """
    Ensure that we set initial_setup_passed condition.
    """
    pbutil.true_cond(status.initial_setup_passed)


def fsm_not_passed(_, spec, status):
    err = Solver.ready_err(spec, status)
    log.info('=== Initial setup not finished: {} ==='.format(err))
    pbutil.false_cond(status.initial_setup.need_reboot, err)
    pbutil.false_cond(status.initial_setup_passed, err)


def fsm_needs_reboot(ctx, _, status):
    # We can enter here several times until reboot is done, no need to spam duplicate logs.
    if not status.initial_setup.boot_id:
        log.info('Setting {} as boot id when initial setup finished'.format(status.node_info.boot_id))
        status.initial_setup.boot_id = status.node_info.boot_id
    # Update status: all done - waiting for reboot to happen.
    pbutil.false_cond(status.initial_setup_passed, constants.AWAITING_REBOOT_STATUS)
    # Signal reboot manager that we're ready to reboot.
    pbutil.true_cond(status.initial_setup.need_reboot, 'To finish initial setup')


def fsm_reboot_done(ctx, spec, status):
    pbutil.false_cond(status.initial_setup.need_reboot, 'Reboot done')
    # Remove marker
    err = None
    try:
        os.unlink(ctx.setup_marker)
    except EnvironmentError as e:
        if e.errno != errno.ENOENT:
            log.error("Failed to remove '{}': {}".format(ctx.setup_marker, e))
            err = "failed to remove '{}': {}".format(ctx.setup_marker, e)
    if err is not None:
        pbutil.false_cond(status.initial_setup_passed, err)
    else:
        log.info('=== Assuming initial setup has finished ===')
        spec.need_initial_setup = False
        pbutil.true_cond(status.initial_setup_passed)


def infer_status(spec, status):
    if spec.need_initial_setup is False:
        return Status.PASSED
    if status.initial_setup.boot_id and status.initial_setup.boot_id != status.node_info.boot_id:
        return Status.REBOOT_DONE
    if Solver.ready_err(spec, status) is None:
        return Status.NEEDS_REBOOT
    return Status.NOT_READY


FSM = {
    Status.PASSED: fsm_passed,
    Status.NOT_READY: fsm_not_passed,
    Status.NEEDS_REBOOT: fsm_needs_reboot,
    Status.REBOOT_DONE: fsm_reboot_done,
}


class Ctx(object):
    def __init__(self, setup_marker):
        self.setup_marker = setup_marker


class InitialSetup(object):
    """
    Manages initial setup - i.e. waits until all states/daemons/etc are executed correctly
    and then signals its readiness to reboot.
    """

    def __init__(self, setup_marker=None):
        self.setup_marker = setup_marker or constants.INITIAL_SETUP_MARKER

    def need_initial_setup(self):
        return os.path.exists(self.setup_marker)

    def run(self, spec, status):
        # Init spec from marker
        spec.need_initial_setup = self.need_initial_setup()
        s = infer_status(spec, status)
        ctx = Ctx(self.setup_marker)
        FSM[s](ctx, spec, status)
