"""
Knows how to manager kernel on host.
"""
import errno
import pkg_resources
import os

from . import grub


def sort_versions(versions):
    versions.sort(key=lambda v: pkg_resources.parse_version(v), reverse=True)
    return versions


class Manager(object):
    """
    For now we only can check current version.
    """
    SERVER_DEB = 'linux-image-server'
    TOOLS_DEB = 'linux-tools'
    GENERIC_DEB = 'linux-image-generic'
    VMLINUZ_LEN = len('vmlinuz-')

    @staticmethod
    def get_version():
        return os.uname()[2]

    @classmethod
    def get_latest_boot_version(cls, boot_dir='/boot'):
        try:
            entries = os.listdir(boot_dir)
        except EnvironmentError as e:
            return None, 'failed to list {}: {}'.format(boot_dir, errno.errorcode.get(e.errno))
        kernels = []
        for entry in entries:
            if entry.startswith('vmlinuz-'):
                kernels.append(entry)
        if not kernels:
            return None, 'no vmlinuz-* files found in {}'.format(boot_dir)
        if len(kernels) == 1:
            entry = kernels[0]
        else:
            entry = sort_versions(kernels)[0]
        return entry[cls.VMLINUZ_LEN:], None

    @classmethod
    def get_boot_version(cls, boot_dir='/boot'):
        """
        Returns kernel version which will be booted in.

        There is no clear way to determine which version will be actually booted.

        So we take logic from salt, which parses grub and lists /boot.
        """
        latest, err = cls.get_latest_boot_version(boot_dir)
        if err is not None:
            return None, err
        g, err = grub.parse_grub_conf(os.path.join(boot_dir, 'grub', 'grub.cfg'))
        if err is not None:
            # Don't know how to handle, let's assume latest for now
            return latest, None
        v, err = g.get_boot_version()
        if err is not None:
            # Same thing
            return latest, None
        return v, None

    def __init__(self, pkg_man):
        self.pkg_man = pkg_man

    def install(self, version):
        pass
