import errno
import os


class ModInfo(object):
    def __init__(self, name, version):
        self.name = name
        self.version = version

    def __str__(self):
        return 'Mod(name={},version={})'.format(self.name, self.version)

    __repr__ = __str__


def _read_version(mod_path):
    """
    Tries to read version file for specified module path.
    """
    p = os.path.join(mod_path, 'version')
    try:
        with open(p) as f:
            buf = f.read(512)
    except EnvironmentError as e:
        if e.errno == errno.ENOENT:
            return 'unknown', None
        return None, str(e)
    return buf.rstrip('\n'), None


def get_mod_info(modname, syspath='/sys'):
    """
    Returns module status.

    :param modname: module name, e.g. nvidia
    :param syspath: prefix to sysfs (for testing purposes mainly)

    :rtype: (ModInfo|None, str|None)
    """
    p = os.path.join(syspath, 'module', modname)
    try:
        if not os.path.isdir(p):
            return None, None
    except EnvironmentError as e:
        return None, str(e)
    v, err = _read_version(p)
    if err is not None:
        return None, err
    return ModInfo(modname, v), None
