"""
Network interfaces related utilities.
"""
import errno
import os
import socket
import logging
import re

import pyroute2

from infra.netconfig.lib import status
from infra.ya_salt.lib import constants

BACKBONE_GROUP_ID = 1


class NetInfo(object):
    def __init__(self,
                 ifname=None,
                 bandwidth=None,
                 bb_fqdn=None, fb_fqdn=None,
                 bb_prefix=None, fb_prefix=None,
                 bb_ipv6_addr=None, fb_ipv6_addr=None):
        self.ifname = ifname
        self.bandwidth = bandwidth
        self.fb_fqdn = fb_fqdn
        self.bb_fqdn = bb_fqdn
        self.bb_prefix = bb_prefix
        self.fb_prefix = fb_prefix
        self.bb_ipv6_addr = bb_ipv6_addr
        self.fb_ipv6_addr = fb_ipv6_addr


def query_ifnames(names_re=re.compile(constants.NET_IFACE_RE), _netdir='/sys/class/net'):
    """
    Queries system interfaces filtering names. Always returns sorted results.
    """
    ifnames = [ifname for ifname in os.listdir(_netdir) if names_re.match(ifname)]
    ifnames.sort()
    return ifnames


def ifname_is_phy(ifname):
    """
    Returns true if ifname represents physical device (has device/vendor).
    """
    return os.path.exists(os.path.join('/sys/class/net', ifname, 'device/vendor'))


def ifname_carrier(ifname):
    """
    Returns true if ifname has carrier - i.e plugged in.
    """
    p = '/sys/class/net/{}/carrier'.format(ifname)
    with open(p) as f:
        try:
            return f.read(128).startswith('1')
        except EnvironmentError as e:
            # If link is administratively disabled, we get EINVAL
            if e.errno == errno.EINVAL:
                return False
            raise


def ifname_speed(ifname, open_fn=open):
    """
    Returns ifname speed according to sysfs.
    """
    with open_fn(os.path.join('/sys/class/net', ifname, 'speed'), 'r') as f:
        return int(f.read().strip())


def ifname_index(ifname):
    """
    Returns ifname index (integer) - to be used in netlink functions.
    """
    p = os.path.join('/sys/class/net', ifname, 'ifindex')
    with open(p) as f:
        return int(f.read(128))


def ifname_is_backbone(ifname):
    """
    Returns true if provided ifname is marked as backbone interface (by netconfig)
    """
    idx = ifname_index(ifname)
    with pyroute2.IPRoute() as ipr:
        link = ipr.link('get', index=idx)[0]
    return link.get_attr('IFLA_GROUP') == BACKBONE_GROUP_ID


def backbone_ifname():
    """
    Queries system filtering network links until we find:
      * eth* named
      * physical interface
      * connected (has carrier)
      * has backbone group attribute
    first one matched is returned.
    """
    try:
        for ifname in query_ifnames(names_re=re.compile(constants.NET_IFACE_RE)):
            if not ifname_is_phy(ifname):
                continue
            if not ifname_carrier(ifname):
                continue
            if not ifname_is_backbone(ifname):
                continue
            return ifname, None
    except Exception as e:
        return None, str(e)
    return None, "no backbone ifname found"


def query_netconfig(net_info):
    """
    Fills net info object using netconfig routines.

    :type net_info: NetInfo
    """
    # Disable netconfig own logging
    logging.disable(logging.CRITICAL)
    try:
        for iface in query_ifnames():
            iface_pb = status.make_iface_proto(iface)
            if not iface_pb:
                continue
            if iface_pb.kind == 'physical' and iface_pb.group == 'backbone':
                net_info.ifname = iface_pb.name
                net_info.bb_ipv6_addr = iface_pb.ipv6_address
                continue
            if iface_pb.kind == 'vlan' and iface_pb.vlan_id == 688:
                net_info.bb_prefix = iface_pb.network
                continue
            if iface_pb.kind == 'vlan' and iface_pb.vlan_id == 788:
                net_info.fb_prefix = iface_pb.network
                continue
            if (iface_pb.kind == 'vlan'
                    and iface_pb.group == 'fastbone'
                    and iface_pb.vlan_id not in status.MTN_VLANS):
                net_info.fb_ipv6_addr = iface_pb.ipv6_address
                continue
    finally:
        # Re-enable logging after netconfig call
        logging.disable(logging.NOTSET)


def get_network_info():
    info = NetInfo()
    query_netconfig(info)

    info.bb_fqdn = socket.gethostname()
    info.fb_fqdn = 'fb-{}'.format(info.bb_fqdn)
    info.bandwidth = ifname_speed(info.ifname)
    return info
