import datetime
import json

from . import base
from infra.ya_salt.lib import statusutil

__all__ = [
    'pformat_apt',
    'pformat_kernel',
    'pformat_run',
    'pformat_salt',
    'pformat_salt_short',
    'pprint_salt_state',
    'pformat_yasm',
    'pformat_exec',
]


def _ts_to_local_dt(ts):
    if not ts.seconds:
        return '-'
    return str(datetime.datetime.fromtimestamp(ts.seconds))


def _uts_to_local_dt(ts):
    return str(datetime.datetime.fromtimestamp(ts))


def _fmt_duration(dur):
    return '{:.3f}s'.format(dur)


def pformat_yasm(status, out):
    out.set_indent(1)
    def_color = 'blue'
    out.write('--- Yasm-Agent-Package')
    if status.agent_package.last_update_ok.status != "True":
        def_color = 'red'

    out.write('  Current-Version: {}'.format(status.agent_package.version or 'unknown'), def_color=def_color)
    out.write('  Last-Update-Ok: {}'.format(status.agent_package.last_update_ok.status == 'True'),
              def_color=def_color)
    out.write('  Last-Transition-Time: {}'.format(
        _ts_to_local_dt(status.agent_package.last_update_ok.transition_time)), def_color=def_color)
    out.write('  Last-Update-Message: "{}"'.format(status.agent_package.last_update_ok.message),
              def_color=def_color)
    out.write('--- Yasm-Agent-Service:')
    if status.agent_service.last_check_ok.status != "True":
        def_color = 'red'

    out.write('  Current-Version: {}'.format(status.agent_service.version or 'unknown'), def_color=def_color)
    out.write('  Last-Check-Ok: {}'.format(status.agent_service.last_check_ok.status == 'True'),
              def_color=def_color)
    out.write('  Last-Transition-Time: {}'.format(
        _ts_to_local_dt(status.agent_service.last_check_ok.transition_time)), def_color=def_color)
    out.write('  Last-Check-Message: "{}"'.format(status.agent_service.last_check_ok.message),
              def_color=def_color)


def pformat_apt(status, out):
    out.set_indent(1)
    def_color = 'blue'
    out.write('--- Apt status')
    if status.last_update_ok.status != "True":
        def_color = 'red'

    out.write('  Last-Update-Ok: {}'.format(status.last_update_ok.status == 'True'), def_color=def_color)
    out.write('  Last-Transition-Time: {}'.format(_ts_to_local_dt(status.last_update_ok.transition_time)),
              def_color=def_color)
    out.write('  Last-Update-Message: "{}"'.format(status.last_update_ok.message), def_color=def_color)
    out.write('  Last-Update-Duration: {}s.'.format(status.last_update_duration.ToSeconds()), def_color=def_color)
    out.write('  Next-Update-Time: {}'.format(_ts_to_local_dt(status.next_update_run_time)),
              def_color=def_color)


def pformat_kernel(status, out):
    out.set_indent(1)
    def_color = 'blue'
    out.write('--- Kernel status')
    if status.need_reboot.status != 'False':
        def_color = 'red'

    out.write('  Current-Version: {}'.format(status.version or 'unknown'), def_color=def_color)
    out.write('  Need-Reboot: {}'.format(status.need_reboot.status == 'True'), def_color=def_color)
    out.write('  Last-Transition-Time: {}'.format(_ts_to_local_dt(status.need_reboot.transition_time)),
              def_color=def_color)
    out.write('  Need-Reboot-Message: "{}"'.format(status.need_reboot.message), def_color=def_color)
    out.write('  Boot-Version: {}'.format(status.boot_version.version or 'unknown'), def_color=def_color)
    out.write('  Boot-Version-Error: {}'.format(status.boot_version.error or '-'), def_color=def_color)
    out.write('  Boot-Version-Changed: {}'.format(_ts_to_local_dt(status.boot_version.transition_time)),
              def_color=def_color)


def pformat_cond(prefix, cond, out, color='blue'):
    if cond.status != "True":
        color = 'red'
    out.write('  {}: {}'.format(prefix, cond.status == 'True'), color)
    out.write('  {}-Message: {}'.format(prefix, cond.message or '-'), color)


def pprint_salt_state(s, out):
    color = "green"
    if s.changes:
        color = "orange"
    if not s.ok:
        color = "red"
    out.write("  ---")
    out.write("  ID: {}".format(s.id), def_color=color)
    out.write("  Started: {}".format(_ts_to_local_dt(s.started)), def_color=color)
    out.write("  Duration: {}".format(_fmt_duration(s.duration / 1000.0)), def_color=color)
    out.write("  OK: {}".format(s.ok), def_color=color)
    out.write("  Comment: {}".format(s.comment), def_color=color)
    if s.changes:
        changes = json.dumps(json.loads(s.changes), indent=3)
        out.write("  Changes:", def_color=color)
        for l in changes.split('\n'):
            out.write("    {}".format(l), def_color=color)


def pformat_salt_summary(status, out):
    ok, changed, fail = statusutil.get_executed_states_results(status.salt_components)
    out.add_n()
    out.write('  Count-Ok: {}'.format(ok), def_color='blue')
    out.write('  Count-Fail: {}'.format(fail), def_color='blue')
    out.write('  Count-Changed: {}'.format(changed), def_color='blue')
    out.write('  Rev: {}'.format(status.salt.revision), def_color='blue')
    out.write('  Rev-Timestamp: {} (ts: {})'.format(_ts_to_local_dt(status.salt.revision_timestamp),
                                                    status.salt.revision_timestamp.ToSeconds()),
              def_color='blue')
    out.write('  Duration: {}'.format(_fmt_duration(status.salt.last_run_duration.ToMilliseconds() / 1000.0)), def_color='blue')
    pformat_cond('Compile-Ok', status.salt.compilation_ok, out)
    pformat_cond('Exec-Ok', status.salt.execution_ok, out)
    out.add_n()


def pformat_salt(status, out):
    out.set_indent(1)
    out.write('--- Salt')

    def _pprint_callback(result):
        return pprint_salt_state(result, out)
    statusutil.enumerate_executed_states(status.salt_components, _pprint_callback)
    pformat_salt_summary(status, out)


def pformat_salt_short(status, out):
    out.set_indent(1)
    out.write('--- Salt')

    def _pprint_callback(result):
        color = "green"
        if result.changes:
            color = "orange"
        if not result.ok:
            color = "red"
        if not result.ok or result.changes:
            out.write(" ID: {}".format(result.id), def_color=color)
            out.write("  Started: {}".format(_ts_to_local_dt(result.started)), def_color=color)
            out.write("  Duration: {}".format(_fmt_duration(result.duration / 1000.0)), def_color=color)
            out.write("  Comment: {}".format(result.comment), def_color=color)
            if result.changes:
                changes = json.dumps(json.loads(result.changes), indent=3)
                out.write("  Changes:", def_color=color)
                for l in changes.split('\n'):
                    out.write("     {}".format(l), def_color=color)
    statusutil.enumerate_executed_states(status.salt_components, _pprint_callback)
    pformat_salt_summary(status, out)


def pformat_run(status, nocolor=False, short=False):
    if nocolor:
        out = base.NoColorOutput()
    else:
        out = base.ColorOutput()
    if short:
        pformat_salt_short(status, out)
    else:
        pformat_salt(status, out)
    pformat_cond('Initial-Setup-Passed', status.initial_setup_passed, out)
    pformat_apt(status.apt, out)
    pformat_kernel(status.kernel, out)
    pformat_yasm(status.yasm, out)
    return str(out)


def pformat_exec(status, nocolor=False, short=False):
    if nocolor:
        out = base.NoColorOutput()
    else:
        out = base.ColorOutput()
    if short:
        pformat_salt_short(status, out)
    else:
        pformat_salt(status, out)
    return str(out)
