"""
Package manager interface and apt-get implementation.
"""
import io
import logging
import os
import shutil
import errno

from infra.ya_salt.lib import subprocutil

DPKG_FIX_CMD = ['/usr/bin/dpkg', '--configure', '-a']
DPKG_FIX_TIMEOUT_SECONDS = 10 * 60
APT_FIX_CMD = ['/usr/bin/apt', 'install', '-f']

log = logging.getLogger('package-manager')


class Metrics(object):
    """
    Package manager metrics counters:
      * installed_ok - number of packages installed
      * install_failures - number of installation failures
      * fix_attempts - number of dpkg fix attempts made
      * fix_failures - number of dpkg fix attempts failures
      * purge_ok - number of packages purged
      * purge_failures - number of purge failures
    """

    def __init__(self):
        self.installed_ok = 0
        self.install_failures = 0
        self.fix_attempts = 0
        self.fix_failures = 0
        self.purged_ok = 0
        self.purge_failures = 0

    def to_yasm(self):
        return [
            ('packages-installed-ok_thhh', self.installed_ok),
            ('packages-install-failures_thhh', self.install_failures),
            ('packages-fix-attempts_thhh', self.fix_attempts),
            ('packages-fix-failures_thhh', self.fix_failures),
        ]


class PackageStatus(object):
    def __init__(self, name, version, installed):
        self.name = name
        self.version = version
        self.installed = installed

    def __str__(self):
        return '(name={},version={},installed={})'.format(self.name,
                                                          self.version,
                                                          self.installed)


class PackageManager(object):
    """
    Interface for package manager.
    """

    def update(self):
        """
        Updates local packages cache and returns error if any or None.

        :rtype: None|unicode
        """
        raise NotImplementedError

    def get_package_status(self, p_name):
        """
        Returns package status and error if failed.

        :rtype: (PackageStatus, None|unicode)
        """
        raise NotImplementedError

    def install(self, p_name, version):
        raise NotImplementedError

    def install_set(self, package_set):
        raise NotImplementedError

    def purge_set(self, names):
        raise NotImplementedError

    def purge(self, p_name):
        raise NotImplementedError

    def repair(self, stderr):
        raise NotImplementedError

    def repair_half_configured(self):
        pass

    def list(self):
        """
        Lists all installed packages on machine.

        :rtype: (list[PackageStatus], None|str)
        """
        raise NotImplementedError

    def metrics(self):
        """
        Get package manager metrics.

        :rtype: Metrics
        """
        raise NotImplementedError


def parse_dpkg_query_out(out):
    parts = out.split('\t', 3)
    if len(parts) != 3:
        return None, 'failed to parse dpkg output "{}"'.format(out)
    return PackageStatus(parts[0], parts[1], parts[2] == 'install ok installed\n'), None


def has_free_space(path):
    try:
        return os.statvfs(path).f_bavail > 0, None
    except EnvironmentError as e:
        return False, str(e)


def get_tmp_dir(envtmp):
    err = 'no suitable temp dirs'
    if envtmp:
        ok, err = has_free_space(envtmp)
        if err is None and ok:
            return envtmp, None
    # Let's try /tmp too
    if envtmp != '/tmp':
        ok, err = has_free_space('/tmp')
        if err is None and ok:
            return '/tmp', None
    return None, err


def is_pkg_not_installed(err):
    """
    Parses dpkg-query output to check if error indicates that package is not installed.
    """
    if err.startswith('dpkg-query: no packages found matching'):
        return True
    if err.startswith('No packages found matching'):
        return True
    return False


class Dpkg(PackageManager):
    # Copied from salt/modules/aptpkg.py
    DPKG_ENV_VARS = {
        'APT_LISTBUGS_FRONTEND': 'none',
        'APT_LISTCHANGES_FRONTEND': 'none',
        'DEBIAN_FRONTEND': 'noninteractive',
        'UCF_FORCE_CONFFOLD': '1',
        # We parse output, ensure locale
        # https://stackoverflow.com/questions/30479607/explain-the-effects-of-export-lang-lc-ctype-and-lc-all/30479702#30479702
        # LC_ALL overrides any other locale-related variable
        'LC_ALL': "C",
        # Put this env (this is documented API, do not remove)
        'HOSTMAN': '1',
    }
    APT_LISTS_DIR = '/var/lib/apt/lists'
    GET_QUERY = ["/usr/bin/dpkg-query", '-W', "-f=${Package}\t${Version}\t${Status}\n"]
    INSTALL_PREFIX = ["/usr/bin/apt-get",
                      "install",
                      "-q", "-y",
                      # These options are taken from salt for compatibility
                      '-o', 'DPkg::Options::=--force-confold',
                      '-o', 'DPkg::Options::=--force-confdef',
                      # 12.04 doesn't have --allow-downgrade, so for now
                      # let's live with this one:
                      "--force-yes",
                      # avoid crazy dependencies installation, see HOSTMAN-554 for more info
                      "--no-install-recommends",
                      ]
    PURGE_PREFIX = ["/usr/bin/apt-get",
                    "purge",
                    "-q", "-y",
                    ]
    APT_GET_TIMEOUT = 10 * 60  # apt-get update run timeout
    APT_GET_UPDATE = ['/usr/bin/apt-get', '-q', '-o' 'Debug::pkgAcquire=1', 'update']
    # In some cases we've seen packages with thousands of files
    # each requires sync which on loaded hosts can take 500ms.
    INSTALL_TIMEOUT_SECONDS = 30 * 60
    SET_TIMEOUT_SECONDS = INSTALL_TIMEOUT_SECONDS * 3
    PURGE_TIMEOUT_SECONDS = 30
    LIST_TIMEOUT_SECONDS = 30

    @classmethod
    def get_env(cls, get_tmp=get_tmp_dir):
        env = os.environ.copy()
        env.update(cls.DPKG_ENV_VARS)
        # Default TMPDIR is currently pointing to /place/vartmp, which is typically
        # consumed by user containers, thus free space can be exhausted.
        # Unfortunately apt saves some files in TMPDIR.
        # To make salt more robust, let's try /tmp too.
        tmpdir, err = get_tmp(env.get('TMPDIR'))
        if err is not None:
            return None, 'cannot get TMPDIR: {}'.format(err)
        # tmpdir can be None if there is no space left, but let's try
        # maybe apt will work
        if tmpdir:
            env['TMPDIR'] = tmpdir
        return env, None

    def __init__(self, check_out=subprocutil.check_output):
        self.check_out = check_out
        self._m = Metrics()

    def repair(self, stderr):
        repair_routine = self._determine_repair_routine(stderr)
        if repair_routine:
            self._m.fix_attempts += 1
            err = repair_routine()
            if err:
                self._m.fix_failures += 1
                return err

    def repair_half_configured(self):
        """
        Tries to fix dpkg by running what it suggests.
        """
        errors = []
        log.info('Fixing dpkg with {}...'.format(' '.join(DPKG_FIX_CMD)))
        env, err = self.get_env()
        if err is not None:
            errors.append(err)
            return errors
        status = subprocutil.check_output(DPKG_FIX_CMD,
                                          timeout=DPKG_FIX_TIMEOUT_SECONDS,
                                          env=env)[2]
        if not status.ok:
            errors.append('dpkg fix failed: {}'.format(status.message))
        else:
            # Seems there is no need for second action
            return None
        log.info('Fixing with {}...'.format(' '.join(APT_FIX_CMD)))
        status = subprocutil.check_output(APT_FIX_CMD,
                                          timeout=DPKG_FIX_TIMEOUT_SECONDS,
                                          env=env)[2]
        if not status.ok:
            errors.append('apt fix failed: {}'.format(status.message))
        return errors

    def _repair_broken_lists(self):
        """
        Tries to fix empty package lists removing all lists
        """
        errors = []
        for i in os.listdir(self.APT_LISTS_DIR):
            p = os.path.join(self.APT_LISTS_DIR, i)
            if os.path.isfile(p) and i != 'lock':
                try:
                    os.unlink(p)
                except OSError as e:
                    if e.errno != errno.ENOENT:
                        errors.append('cannot remove possibly broken list {}: {}'.format(p, e))
        shutil.rmtree(os.path.join(self.APT_LISTS_DIR, 'partial'), True)
        if not errors:
            err = self.update()
            if err:
                errors.append('failed to update lists after fix: {}'.format(err))
        return errors

    @staticmethod
    def _repair_unknown():
        return ('cannot repair unknown apt/dpkg problem',)

    def _determine_repair_routine(self, stderr):
        if 'dpkg was interrupted, you must manually run' in stderr:
            return self.repair_half_configured
        elif 'package lists or status file could not be parsed or opened' in stderr:
            return self._repair_broken_lists
        else:
            return self._repair_unknown

    def _run_dpkg_command(self, cmdline, timeout, env=None):
        """
        Runs package management command, trying to fix and restart it if needed.
        """
        if env is None:
            env, err = self.get_env()
            if err is not None:
                return None, err, subprocutil.Status(False, err)
        log.debug('Running command (TMPDIR={}): {}'.format(env.get('TMPDIR') or '<unset>', cmdline))
        out, err, status = self.check_out(cmdline, timeout=timeout, env=env)
        if not status.ok:
            log.warning('Running command {} failed. Trying to fix that.'.format(cmdline))
            e = self.repair(err)
            if not e:
                return self.check_out(cmdline, timeout=timeout, env=env)
            else:
                log.error('Failed to apply fixes for {}: {}'.format(cmdline, e))
        return out, err, status

    def update(self):
        env, err = self.get_env()
        if err is not None:
            return err
        stdout, stderr, status = self._run_dpkg_command(self.APT_GET_UPDATE,
                                                        self.APT_GET_TIMEOUT,
                                                        env=env)
        if status.ok:
            return None
        # log apt-get update output if update failed
        log.error("apt-get update failed stdout: {}".format(stdout))
        log.error("apt-get update failed stderr: {}".format(stderr))
        return '; '.join([status.message, subprocutil.tail_of(stderr, prefix='stderr: ')])

    def get_package_status(self, p_name):
        cmdline = self.GET_QUERY[:]
        cmdline.append(p_name)
        out, err, status = self._run_dpkg_command(cmdline, timeout=30)
        if not status.ok:
            if err is not None and is_pkg_not_installed(err):
                return PackageStatus(p_name, 'unknown', False), None
            return None, 'failed to query version for "{}": {}'.format(p_name, status.message)
        return parse_dpkg_query_out(out)

    def install(self, p_name, version):
        pkg_str = '{}={}'.format(p_name, version)
        cmdline = self.INSTALL_PREFIX[:]
        cmdline.append(pkg_str)
        status = self._run_dpkg_command(cmdline, timeout=self.INSTALL_TIMEOUT_SECONDS)[2]
        if not status.ok:
            self._m.install_failures += 1
            return None, 'failed to install {}: {}'.format(pkg_str, status.message)
        self._m.installed_ok += 1
        return PackageStatus(p_name, version, True), None

    def install_set(self, package_set):
        cmdline = self.INSTALL_PREFIX[:]
        n = 0
        for name, version in package_set:
            pkg_str = '{}={}'.format(name, version)
            cmdline.append(pkg_str)
            n += 1
        status = self._run_dpkg_command(cmdline, timeout=self.SET_TIMEOUT_SECONDS)[2]
        if not status.ok:
            self._m.install_failures += n
            return status.message
        self._m.installed_ok += n
        return None

    def purge(self, p_name):
        cmdline = self.PURGE_PREFIX[:]
        cmdline.append(p_name)
        out, err, status = self._run_dpkg_command(cmdline,
                                                  timeout=self.PURGE_TIMEOUT_SECONDS)
        if not status.ok:
            self._m.purge_failures += 1
            return 'failed to purge {}: {}'.format(p_name, status.message)
        self._m.purged_ok += 1
        return None

    def purge_set(self, names):
        cmdline = self.PURGE_PREFIX[:]
        cmdline.extend(names)
        out, err, status = self._run_dpkg_command(cmdline,
                                                  timeout=self.PURGE_TIMEOUT_SECONDS)
        if not status.ok:
            self._m.purge_failures += 1
            return 'failed to purge {}: {}'.format(', '.join(names), status.message)
        self._m.purged_ok += 1
        return None

    def list(self):
        cmdline = self.GET_QUERY[:]
        out, err, status = self._run_dpkg_command(cmdline,
                                                  timeout=self.LIST_TIMEOUT_SECONDS)
        if not status.ok:
            return None, 'failed to list packages: {}'.format(status.message)
        rv = []
        for l in io.BytesIO(out):
            p, err = parse_dpkg_query_out(l)
            if err is not None:
                return None, err
            rv.append(p)
        return rv, None

    def metrics(self):
        return self._m
