import json
import hashlib
import os

from google.protobuf import json_format
import yaml

from infra.ya_salt.lib import fileutil


def set_condition(cond, status, message='OK'):
    if cond.status != status:
        cond.transition_time.GetCurrentTime()
    cond.status = status
    cond.message = message


def false_cond(cond, message):
    set_condition(cond, 'False', message)


def true_cond(cond, message='OK'):
    set_condition(cond, 'True', message)


def unkn_cond(cond, message):
    set_condition(cond, 'Unknown', message)


def update_status_from_result(results, salt_status):
    for d in results.values():
        # See tests for structure examples
        r = salt_status.state_results.add()
        r.sls = d.get('__sls__') or 'unknown'
        if 'result' in d:
            r.ok = bool(d['result'])
            r.comment = d.get('comment', '')
            # Not quite sure what to use as id, let's try all options
            try:
                r.id = d['__id__']
            except KeyError:
                r.id = d.get('name') or d.get('__sls__') or '<unknown>'
                continue
            r.duration = d.get('duration', 0)
            r.started.FromSeconds(d['started'])
            if 'changes' in d and d['changes']:
                r.changes = json.dumps(d['changes'])


def pb_from_dict(m, d):
    try:
        json_format.ParseDict(d, m, ignore_unknown_fields=True)
    except Exception as e:
        return str(e)
    return None


def pb_to_dict(m):
    return json_format.MessageToDict(m)


def pb_to_json(m):
    # also dump default values, used for debug output in nodeinfo
    return json_format.MessageToJson(m, including_default_value_fields=True)


def meta_from_dict(meta, d):
    d_meta = d.get('meta')
    if d_meta is None:
        return 'no meta field'
    if not isinstance(d_meta, dict):
        return 'meta field is not a mapping'
    # Unmarshal object meta data
    err = pb_from_dict(meta, d_meta)
    if err is not None:
        return err
    if not meta.kind:
        return 'no kind field'
    return None


def spec_from_dict(spec, d):
    # Unmarshal object spec
    d_spec = d.get('spec')
    if d_spec is None:
        return 'no spec field'
    if not isinstance(d_spec, dict):
        return 'spec field is not a mapping'
    err = pb_from_dict(spec, d_spec)
    if err is not None:
        return err
    return None


def obj_from_dict(meta, spec, d):
    err = meta_from_dict(meta, d)
    if err is not None:
        return err
    if meta.kind != spec.DESCRIPTOR.name:
        return "invalid object kind: got='{}', want='{}'".format(meta.kind, spec.DESCRIPTOR.name)
    return spec_from_dict(spec, d)


def obj_from_buf(meta, spec, fmt, buf):
    d, err = dict_from_buf(buf, fmt)
    if err is not None:
        return err
    return obj_from_dict(meta, spec, d)


def dict_from_buf(buf, fmt):
    if fmt == 'json':
        try:
            d = json.loads(buf)
        except Exception as e:
            return None, 'failed to parse as JSON: {}'.format(e)
    elif fmt in ['yaml', 'yml']:
        try:
            d = yaml.load(buf, Loader=yaml.SafeLoader)
        except Exception as e:
            return None, 'failed to parse as YAML: {}'.format(e)
    else:
        return None, 'unsupported format: {}'.format(fmt)
    if not isinstance(d, dict):
        return None, 'content is not a mapping'
    return d, None


def spec_from_file(meta, spec, path, maxsize=1 * 1024 * 1024):
    # Figure out format from file name extension
    ext = os.path.splitext(path)[1]
    meta.annotations['filename'] = path
    if not ext:
        return "file '{}' has no extension".format(path)
    fmt = ext[1:]
    if not fmt:
        return "file '{}' extension is empty".format(path)
    # Read file and parse it into a mapping
    buf, err = fileutil.read_file(path, maxsize)
    if err is not None:
        return err
    return obj_from_buf(meta, spec, fmt, buf)


def pb_from_file(m, path):
    # Figure out format from file name extension
    ext = os.path.splitext(path)[1]
    if not ext:
        return "file '{}' has no extension".format(path)
    fmt = ext[1:]
    if not fmt:
        return "file '{}' extension is empty".format(path)
    # Read file and parse it into a mapping
    buf, err = fileutil.read_file(path)
    if err is not None:
        return err
    d, err = dict_from_buf(buf, fmt)
    if err is not None:
        return err
    return pb_from_dict(m, d)


def pb_digest(m):
    return hashlib.sha1(m.SerializeToString(deterministic=True)).hexdigest()
