"""
Manages persistence related features.
"""
from infra.ya_salt.lib import constants
from infra.ya_salt.lib import fileutil
from infra.ya_salt.proto import ya_salt_pb2


class Marshaller(object):
    MAX_SIZE = 5 * 1024 * 1024
    MOD = 0644

    def __init__(self, path, message_cls, max_size=MAX_SIZE, mod=MOD):
        self.path = path
        self.message_cls = message_cls
        self.max_size = max_size
        self.mod = mod

    def load(self):
        buf, err = fileutil.read_file(self.path, self.max_size)
        if err is not None:
            return None, err
        m = self.message_cls()
        try:
            m.MergeFromString(buf)
        except Exception as e:
            return None, 'failed to deserialize "{}": {}'.format(self.path, e)
        return m, None

    def save(self, m):
        buf = m.SerializeToString()
        err = fileutil.atomic_write(self.path, buf, chmod=self.mod,
                                    make_dirs=True)
        if err is not None:
            return 'failed to save state: {}'.format(err)
        return None


def load_status(path=constants.STATUS_PATH):
    return Marshaller(path, ya_salt_pb2.HostmanStatus).load()


def save_status(m):
    return Marshaller(constants.STATUS_PATH, ya_salt_pb2.HostmanStatus).save(m)


def load_spec(path=constants.SPEC_PATH):
    return Marshaller(path, ya_salt_pb2.HostmanSpec).load()


def save_spec(m):
    return Marshaller(constants.SPEC_PATH, ya_salt_pb2.HostmanSpec).save(m)
