"""
Ex-grain gathering some info about mount points.
This module is used *at least* by yasm (yasmagent/core/tagsparser.py).
"""
import os
import subprocess

DEFAULT_DIR = '/place'
SSD_DIR = '/ssd'
YT_PATH = '/yt'


def _check_disk_type(disk):
    name = disk.split('/')[-1]
    with open('/sys/block/{}/queue/rotational'.format(name)) as file_desc:
        check_ssd = int(file_desc.read())
    if check_ssd == 0:
        if name.startswith('nvme'):
            return "nvme"
        return "ssd"
    else:
        return "hdd"


def _get_mount_point(path):
    # realpath motivation is as follows:
    # `path` argument here means one of {DEFAULT,SSD}_PATH,YT_PATH
    # `/ssd` can actually be symlink to `/place`, which is a mount point.
    # If we do not resolve symlink - we get `/` which is not correct.
    path = os.path.realpath(path)
    while not os.path.ismount(path):
        path = os.path.dirname(path)
    return path


def _get_device(path):
    devices = []
    with open('/proc/mounts') as file_desc:
        info = file_desc.readlines()
    for line in info:
        line_arr = line.split()
        if path == line_arr[1]:
            devices.append(line_arr[0])
    if devices[0] == 'rootfs':
        return os.path.realpath(devices[1])
    if devices[0].startswith('/dev/mapper/'):
        return devices[0]
    return os.path.realpath(devices[0])


def _get_raid_info(device):
    # disks = list_disks()
    mddev = device.split('/')[-1]
    with open('/proc/mdstat') as mdstat:
        for line in mdstat.readlines():
            if not line.startswith('md'):
                continue
            line_arr = line.split()
            if line_arr[0] != mddev:
                continue
            # md_name = line_arr[0]
            md_type = line_arr[3]
            md_partisipant = []
            # md_disks_type = []
            for disk in line_arr[4:]:
                if disk.startswith('sd'):
                    disk = disk[:4]
                if disk.startswith('nvme'):
                    disk = disk[:8]
                md_partisipant.append('/dev/' + disk)
                # if disks[disk[:3]]['ssd'] not in md_disks_type:
                #     md_disks_type.append(disks[disk[:3]]['ssd'])
        return md_type, md_partisipant


def _get_lvm_info(device):
    cmd = 'lvs -o devices,lv_attr --aligned --noheadings {}'.format(device)
    out = subprocess.check_output(cmd, shell=True)
    disks = []
    disk_type = None
    for line in out.split('\n'):
        line = line.strip()
        if line == '':
            continue
        (disk, info) = line.split()
        if info[0] in ['r', 'R']:
            if info[6] == 'm':
                disk_type = 'raid1'
            else:
                disk_type = 'raid0'
        disk = disk.split('(')[0]
        disks.append(disk)
    return disk_type, disks


def _get_path_info(path):
    (mount_point, dev_point, raid_type, disks_info) = (None, None, None, [])
    mount_point = _get_mount_point(path)
    dev_point = _get_device(mount_point)
    if dev_point.split('/')[-1].startswith('md'):
        (raid_type, disks) = _get_raid_info(dev_point)
        for disk in disks:
            if disk.startswith('/dev/sd'):
                disk = disk[5:8]
            if disk.startswith('/dev/nvme'):
                disk = disk[5:12]
            disk_type = _check_disk_type(disk)
            disks_info.append({disk: {'disk_type': disk_type}})
    # elif dev_point.split('/')[-1].startswith('sd'):
    elif dev_point.startswith('/dev/mapper/'):
        (raid_type, disks) = _get_lvm_info(dev_point)
        for disk in disks:
            if disk.startswith('/dev/sd'):
                disk = disk[5:8]
            if disk.startswith('/dev/nvme'):
                disk = disk[5:12]
            disk_type = _check_disk_type(disk)
            disks_info.append({disk: {'disk_type': disk_type}})
    else:
        if dev_point.startswith('/dev/sd'):
            disk = dev_point[5:8]
        elif dev_point.startswith('/dev/nvme'):
            disk = dev_point[5:12]
        disk_type = _check_disk_type(disk)
        disks_info.append({disk: {'disk_type': disk_type}})
    return mount_point, dev_point, raid_type, disks_info


def get():
    """
    Return info about disks in special points
    """
    points_info = {}
    if os.path.exists(DEFAULT_DIR):
        (mount_point, dev_point, raid_type, disks_info) = _get_path_info(DEFAULT_DIR)
        points_info[DEFAULT_DIR] = {
            'mount_point': mount_point,
            'device': dev_point,
            'raid': raid_type,
            'disks_info': disks_info
        }

    if os.path.exists(SSD_DIR):
        (mount_point, dev_point, raid_type, disks_info) = _get_path_info(SSD_DIR)
        points_info[SSD_DIR] = {
            'mount_point': mount_point,
            'device': dev_point,
            'raid': raid_type,
            'disks_info': disks_info
        }
    if os.path.isdir(YT_PATH):
        for f in os.listdir(YT_PATH):
            path_f = os.path.join(YT_PATH, f)
            if not os.path.isdir(path_f):
                continue
            if os.path.islink(path_f):
                continue

            (mount_point, dev_point, raid_type, disks_info) = _get_path_info(path_f)
            points_info[path_f] = {
                'mount_point': mount_point,
                'device': dev_point,
                'raid': raid_type,
                'disks_info': disks_info
            }
    return points_info
