import logging

log = logging.getLogger('reboots')


class Constraint(object):
    """Interface for reboot constraints"""

    def can_reboot(self):
        """
        :rtype: unicode|None
        """
        raise NotImplementedError


class GroupConstraint(Constraint):
    NON_REBOOT_GROUPS = [
        'SAS_NANNY_HQ',
        'MAN_NANNY_HQ',
        'MSK_NANNY_HQ',
        'VLA_SG_CLUSTER_HQ',
        'ALL_COPIER_TRACKER',
        'ALL_STOUT',
        'MAN_ISS3_PRODUCTION',
        'MSK_ISS3_PRODUCTION',
        'SAS_ISS3_PRODUCTION',
        'VLA_ISS3_PRODUCTION',
        'ALL_ISS_CASSANDRA_GLOBAL',
        'ALL_NMT',
    ]

    def __init__(self, groups):
        self.groups = groups

    def can_reboot(self):
        if len(set(self.groups) & set(self.NON_REBOOT_GROUPS)) > 0:
            return 'gencfg group restricts reboot'
        return None


class ProjectConstraint(Constraint):
    YP_DEV_PROJECTS = [
        'yp-iss-sas-dev',
        'yp-iss-man-dev',
        'yp-iss-vla-dev',
    ]
    YT_PROJECTS = [
        # 'rtc-yt-pythia-masters'
    ]
    NON_REBOOT_PROJECTS = YP_DEV_PROJECTS + YT_PROJECTS

    NON_REBOOT_TAGS = ['special_reboot']

    PRJ_ERR = 'walle project restricts reboots'
    TAG_ERR = 'walle project has restricting tag'

    def __init__(self, prj, tags):
        self.prj = prj
        self.tags = tags

    def can_reboot(self):
        if self.prj in self.NON_REBOOT_PROJECTS:
            return self.PRJ_ERR
        if len(set(self.tags) & set(self.NON_REBOOT_TAGS)) > 0:
            return self.TAG_ERR
        return None


class StaticConstraint(Constraint):
    def __init__(self, err):
        self.err = err

    def can_reboot(self):
        return self.err


class WalleConstraint(Constraint):
    """
    :type walle: walle_api.WalleClient|mock.Mock
    """
    PROJECT_DISABLED_MSG = 'project healing automation disabled'
    HOST_DISABLED_MSG = 'host has restrictions in walle'

    TESTING_MANAGED_PROJECTS = ['search-testing',
                                'search-yt-testing',
                                'search-testing-rest',
                                ]

    def __init__(self, walle, hostname, project):
        self.walle = walle
        self.hostname = hostname
        self.project = project

    def can_reboot(self):
        # Allow reboots for hosts in projects managed by testing WALL-E.
        if self.project in self.TESTING_MANAGED_PROJECTS:
            return None
        try:
            d = self.walle.get_host(self.hostname, fields=("restrictions",))
        except Exception as e:
            return 'failed to check walle host restrictions: {}'.format(e)
        if 'restrictions' in d:
            return self.HOST_DISABLED_MSG
        try:
            d = self.walle.get_project(self.project, fields=("healing_automation",))
        except Exception as e:
            return 'failed to check walle project restrictions: {}'.format(e)
        auto_enabled = d.get('healing_automation', {}).get('enabled')
        if not auto_enabled:
            return self.PROJECT_DISABLED_MSG
        return None


def make_orly_rule(prefix, env_type):
    if not env_type:
        # To be on the safe side
        env_type = 'production'
    return prefix + '-' + env_type


class OrlyConstraint(Constraint):
    def __init__(self, orly, rule_id):
        self.orly = orly
        self.rule_id = rule_id

    def can_reboot(self):
        # Check orly rule
        if self.orly is None:
            return None
        err = self.orly.start_operation(self.rule_id)
        if err is not None:
            return err
        return None


class ChainedConstraint(Constraint):
    """
    :type constraints: list[Constraint]
    """

    def __init__(self, constraints):
        self.constraints = constraints

    def can_reboot(self):
        for c in self.constraints:
            err = c.can_reboot()
            if err is not None:
                return err
        return None


class RebootManager(object):
    """
    Entity which infers reboot requests from components.

    Motivation is simple:
      * we do not want components to know how to perform reboots
      * we do not want to reboot (i.e. kill self) during execution
    This approach however has drawbacks. Main drawback is pretty straightforward: async.
    Thus users must work like this:
      * during execution take a look at your status (or env) and decide to reboot
      * set some condition (e.g. kernel.need_reboot)
      * during next execution (which may or may not happen after reboot) - do the same:
        take a look around and do/don't schedule reboot
    Reboot request can fail for several reasons:
      * we might be killed during execution
      * orly can forbid reboot
      * some unexpected error might make us exit early (unhandled exception)
    """

    def __init__(self, orly, powerman):
        self.orly = orly
        self.powerman = powerman

    def run(self, spec, status):
        # Separate cases for initial setup.
        # Initial setup knows about kernel needs, so we do not check kernel branch
        # if initial setup has not passed yet.
        if spec.need_initial_setup:
            if status.initial_setup.need_reboot.status == 'True':
                c = OrlyConstraint(self.orly, make_orly_rule('hostman-initial', spec.env_type))
                reason = status.initial_setup.need_reboot.message
            else:
                # Initial setup not finished, nothing to do
                return
        else:
            # Nothing to do
            return
        err = c.can_reboot()
        if err is not None:
            log.error('Reboot is not allowed: {}'.format(err))
        else:
            log.info('Requesting reboot: {}...'.format(reason))
            err = self.powerman.reboot()
            if err is not None:
                log.error('Reboot failed: {}'.format(err))
                return
