import time
import errno
import fcntl
import os

from infra.ya_salt.lib import constants


def lock_run(path):
    # This fd is **intentionally leaked** to ensure we don't
    # close it during all script run time.
    fd = None
    try:
        os.makedirs(os.path.dirname(path))
    except EnvironmentError as e:
        if e.errno != errno.EEXIST:
            raise
    # Here we do not set FD_CLOEXEC, so potentially
    # this fd can leak into children (which are usually killed upon our death).
    # But grandchildren will potentially inherit this descriptor.
    # Is this right behaviour? That depends:
    #   * At first glance - it it not hygienic
    #   * But! We do not launch long running services and if such fd leaks into
    #   long running service and block our execution - this *IS* an error
    #   and must be debugged.
    try:
        fd = os.open(path, os.O_CREAT)
        fcntl.flock(fd, fcntl.LOCK_NB | fcntl.LOCK_EX)
        return True, fd
    except EnvironmentError as e:
        if fd is not None:
            os.close(fd)
        if e.errno != errno.EWOULDBLOCK:
            raise
        return False, None


class Flag(object):
    """
    Disk based flag with expiration time.
    """
    def __init__(self, path=constants.FILE_DISABLE_RUN, expire_time=constants.TIMEOUT_DISABLE):
        self._path = path
        self._expire_time = expire_time

    def get_path(self):
        return self._path

    def get_expire_time(self):
        return self._expire_time

    def is_enforced(self):
        """
        Checks if flag is enforced. Returns tuple:
          * if flag is enforced
          * if flag file exists (so that user can decide to remove it)
        """
        try:
            st = os.stat(self._path)
        except EnvironmentError as e:
            # File does not exists
            if e.errno == errno.ENOENT:
                return False, False
            # For now don't handle other errors
            raise
        return st.st_mtime + constants.TIMEOUT_DISABLE > time.time(), True

    def remove(self):
        """
        Removes flag if it exists.
        """
        try:
            os.unlink(self._path)
        except EnvironmentError as e:
            if e.errno != errno.ENOENT:
                return str(e)
        return None

    def touch(self, _utime=os.utime):
        """
        Creates flag file on filesystem and adjusts expire time if it exists.
        Returns tuple:
            * if flag existed before and we adjusted mtime
            * error if some action failed
        """
        e = None
        try:
            fd = os.open(self._path, os.O_CREAT | os.O_EXCL)
        except EnvironmentError as e:
            pass
        else:
            os.close(fd)
        if e is None:  # Created file
            return False, None
        elif e.errno == errno.EEXIST:
            # Update mtime in case file existed before -> thus adjust expire time.
            try:
                _utime(self._path, None)
            except Exception as e:
                return False, str(e)
            return True, None
        else:
            return False, str(e)
