"""
Process management utilities.
"""
import os
import signal
import contextlib

import subprocess32
import prctl


@contextlib.contextmanager
def noop_pipe(*args, **kwargs):
    yield subprocess32.PIPE


def tail_of(buf, l=100, prefix=''):
    cut = len(buf) - l + 3  # Ellipsis '...'
    if cut <= 0:
        return prefix + buf
    return prefix + '...' + buf[cut:]


def set_pdeath_sig():
    # Kill executable upon parent death
    prctl.set_pdeathsig(signal.SIGKILL)


class Status(object):
    __slots__ = ('ok', 'message')

    def __init__(self, ok=None, message=None):
        self.ok = ok
        self.message = message


def check_output(argv, timeout, env=None, data=None):
    """
    Run executable in subprocess returning stdout, stderr and status.

    Uses small buffer and should no be used for large outputs or streaming.

    :param list[str] argv: arguments list
    :param int|float timeout: timeout to wait for process output.
    :param str data: data to pass to stdin
    :rtype: (str, str, Status)
    """
    status = Status()
    cmd = ' '.join(argv)
    stdout = ''
    stderr = ''
    if data:
        open_func = noop_pipe
    else:
        open_func = open
    with open_func(os.devnull, 'w') as stdin:
        try:
            p = subprocess32.Popen(argv,
                                   cwd='/',
                                   stdout=subprocess32.PIPE,
                                   stdin=stdin,
                                   stderr=subprocess32.PIPE,
                                   close_fds=True,
                                   preexec_fn=set_pdeath_sig,
                                   bufsize=4096,
                                   env=env,
                                   )
        except Exception as e:
            status.ok = False
            status.message = 'failed to execute "{}": {}'.format(cmd, e)
            return stdout, stderr, status
        try:
            stdout, stderr = p.communicate(timeout=timeout, input=data)
        except subprocess32.TimeoutExpired as e:
            status.ok = False
            status.message = '"{}" execution timed out after {}s'.format(cmd, timeout)
            return e.output or '', stderr, status
        except Exception as e:
            status.ok = False
            status.message = '"{}" execution failed: {}'.format(cmd, e)
        finally:
            if p.returncode is not None:
                p.kill()
                p.wait(30)  # Do not catch this one, let it raise and die
        if p.returncode != 0:
            status.ok = False
            status.message = '"{}" execution failed with exit code: {}'.format(cmd, p.returncode)
        else:
            status.ok = True
        # Force returning strings instead of None
        return stdout or '', stderr or '', status
