import os
import errno

import yaml

from infra.ya_salt.proto import ya_salt_pb2

from infra.ya_salt.lib import fileutil
from infra.ya_salt.lib import pbutil

kinds = {}


def unit_from_path(path):
    """
    Loads unit (meta, spec) from specified **YAML** file. To save some tabbing for user,
    we accept either one document (with spec, meta).
    Thus file can look like:
    meta:
        ...
    spec:
        ...
    """
    buf, err = fileutil.read_file(path, max_size=1 * 1024 * 1024)
    if err is not None:
        return None, None, err
    try:
        obj_yaml = yaml.load(buf, Loader=yaml.CSafeLoader)
    except Exception as e:
        return None, None, 'failed to load YAML: {}'.format(e)
    meta = ya_salt_pb2.ObjectMeta()
    err = pbutil.meta_from_dict(meta, obj_yaml)
    if err is not None:
        return None, None, err
    if not meta.name:
        return meta, None, 'no object name given'
    # Get object spec type from meta document
    pb = kinds.get(meta.kind)
    if pb is None:
        return meta, None, "unknown object kind: '{}'".format(meta.kind)
    spec = pb()
    err = pbutil.spec_from_dict(spec, obj_yaml)
    if err is not None:
        return meta, None, err
    return meta, spec, None


def read_units_from_dir(log, path):
    """
    WARN: NOT USED YET
    Reads unit objects from specified directory.
    """
    units = {}  # name => (meta, spec)
    try:
        entries = os.listdir(path)
    except EnvironmentError as e:
        if e.errno == errno.ENOENT:
            return units, None
        if e.errno == errno.ENOTDIR:
            return None, '{} is not a directory'.format(path)
        return [], "failed to read '{}': {}".format(path, e)
    for entry in entries:
        f_path = os.path.join(path, entry)
        if not os.path.isfile(f_path):
            continue
        meta, spec, err = unit_from_path(f_path)
        if err is not None:
            # Unknown object, maybe do not have even a name
            log.error("Skipping '{}': {}".format(f_path, err))
            continue
        if not meta.name:
            # Unnamed object, cannot match to anything
            log.error("Skipping '{}': no meta.name specified".format(f_path))
        units[meta.name] = (meta, spec)
    return units, None
