import os
import logging
import requests
import time
import json

from infra.ya_salt.lib import constants
from infra.ya_salt.lib import fileutil

log = logging.getLogger('walle')

LOCATION_MSK = ('iva', 'ugrb', 'myt')
RAW_FIELDS = ('country', 'city', 'datacenter', 'queue', 'rack')
CACHE_TIME = 15 * 60  # 15 minutes see HOSTMAN-1157 details
INFO_FILE = os.path.join(constants.VAR_LIB, 'walle_info.txt')
NEED_RE_SYNC_ERR = 'local file needs re sync'


def info_from_http(d):
    """
    :type d: dict
    """
    loc = d.get('location') or {}
    info = {
        'project': d.get('project', ''),
        'tags': d.get('tags'),
        'dc': loc.get('short_datacenter_name', ''),
        'country': loc.get('country', '').lower(),
        'queue': loc.get('short_queue_name', ''),
        'rack': loc.get('rack', ''),
        'switch': loc.get('switch', '')
    }
    if info['dc'] in LOCATION_MSK:
        info['location'] = 'msk'
    else:
        info['location'] = info['dc']
    for field in RAW_FIELDS:
        raw_field = 'raw_{}'.format(field)
        info[raw_field] = loc.get(field, '')
    return info


def validate_info(info):
    prj = info.get('project')
    if prj is None or prj == "":
        return 'no or empty project'
    if not isinstance(prj, basestring):
        return "project '{}' is not a string".format(prj)
    tags = info.get('tags')
    if not tags:
        return 'no or empty tags'
    if not isinstance(tags, list):
        return 'tags is not a list'
    for i, t in enumerate(tags):
        if not isinstance(t, basestring):
            return "tag '{}' at pos {} is not a string".format(t, i)
        if not t:
            return 'empty tag at pos {}'.format(i)
    if 'location' not in info:
        return 'no location'
    loc = info.get('location')
    if not isinstance(loc, basestring):
        return "location '{}' is not a string".format(loc)
    if not loc:
        return 'empty location'
    sw = info.get('switch')
    if not sw:
        return 'no switch'
    if not isinstance(sw, basestring):
        return "switch '{}' is not a string".format(sw)
    return None


def save_info(info):
    """
    Write json to file (cache)
    """
    buf = json.dumps(info, indent=4)
    return fileutil.atomic_write(INFO_FILE, buf, make_dirs=True)


def load_from_fs(path=INFO_FILE, cache_time=CACHE_TIME):
    """
    Checks if file needs update - i.e. request walle.
    """
    now = time.time()
    try:
        with open(path, 'r') as f:
            mtime = os.fstat(f.fileno()).st_mtime
            info = json.load(f)
    except Exception as e:
        return None, str(e)
    if now - mtime > cache_time:
        return info, NEED_RE_SYNC_ERR
    return info, None


def request_remote_info(state):
    """
    Requests server information from WALL-E.
    """
    path = 'https://api.wall-e.yandex-team.ru/v1/hosts/'
    req = requests.Request('GET', os.path.join(path, state),
                           params={
                               'fields': 'project,location,tags',
                               'resolve_tags': True}).prepare()
    headers = {'User-Agent': 'YA-SALT RTC'}
    try:
        # 10sec timeout (HOSTMAN-1105)
        resp = requests.get(req.url, headers=headers, timeout=10)
    except Exception as e:
        return None, str(e)
    if not resp.ok:
        return None, 'bad response code {} {}'.format(resp.status_code, resp.reason)
    try:
        d = resp.json()
    except Exception as e:
        return None, 'failed to get json from walle API: {}'.format(e)
    return info_from_http(d), None


def sync(h):
    info, err = request_remote_info(h)
    if err is not None:
        return None, err
    err = validate_info(info)
    if err is not None:
        log.error('Failed to validate WALL-E response: {}'.format(err))
        return None, err
    err = save_info(info)
    if err is not None:
        log.error('Failed to save remote info: {}'.format(err))
    return info, None


def grains_from_info(info):
    """
    Construct grains from format stored on disk.
    """
    return {
        'walle_project': info['project'],
        'walle_tags': info['tags'],
        'walle_location': info['location'],
        'walle_dc': info['dc'],
        'walle_country': info['country'],
        'walle_queue': info.get('queue', ''),
        'walle_rack': info.get('rack', ''),
        'walle_switch': info.get('switch', ''),
        'location': info['location']
    }


def grains(h, load_fun=load_from_fs):
    info, err = load_fun()
    if err is not None:
        log.info('Requesting WALL-E host info: {}...'.format(err))
        info, err = sync(h)
        if err is not None:
            log.error('Sync failed: {}'.format(err))
        # No return here to live with existing data for static stability
        # in case of WALL-E failure.
    if info is not None:
        return grains_from_info(info), None
    return None, err
