import collections
import yaml
import yaml.constructor
import yaml.resolver
import salt.utils.odict


class MyDumper(yaml.SafeDumper):
    def increase_indent(self, flow=False, indentless=False):
        return super(MyDumper, self).increase_indent(flow, False)


def ordered_dict_representer(dumper, data):
    return dumper.represent_mapping(yaml.resolver.BaseResolver.DEFAULT_MAPPING_TAG, data.items())


MyDumper.add_representer(salt.utils.odict.OrderedDict, ordered_dict_representer)


def prettify(d):
    """
    Formats dictionary as yaml.
    """
    return yaml.dump(d, Dumper=MyDumper, default_flow_style=False, indent=2)


def construct_yaml_map(self, node):
    data = collections.OrderedDict()
    yield data
    value = self.construct_mapping(node)
    data.update(value)


def construct_mapping(self, node, deep=False):
    if isinstance(node, yaml.MappingNode):
        self.flatten_mapping(node)
    else:
        msg = 'Expected a mapping node, but found {}'.format(node.id)
        raise yaml.constructor.ConstructError(None, None, msg, node.start_mark)

    mapping = collections.OrderedDict()
    for key_node, value_node in node.value:
        key = self.construct_object(key_node, deep=deep)
        try:
            hash(key)
        except TypeError as err:
            raise yaml.constructor.ConstructError('while constructing a mapping', node.start_mark,
                                                  'found unacceptable key ({})'.format(err),
                                                  key_node.start_mark)
        value = self.construct_object(value_node, deep=deep)
        mapping[key] = value
    return mapping


class OrderedLoaderMixin(object):
    def __init__(self, *args, **kwargs):
        super(OrderedLoaderMixin, self).__init__(*args, **kwargs)
        self.add_constructor('tag:yaml.org,2002:map', type(self).construct_yaml_map)
        self.add_constructor('tag:yaml.org,2002:omap', type(self).construct_yaml_map)

    construct_yaml_map = construct_yaml_map
    construct_mapping = construct_mapping


class OrderedSafeLoader(OrderedLoaderMixin, yaml.SafeLoader):
    pass


class CLoader(OrderedLoaderMixin, yaml.CLoader):
    pass


class CSafeLoader(OrderedLoaderMixin, yaml.CSafeLoader):
    pass
