package main

import (
	"bytes"
	"encoding/json"
	"fmt"
	"io"
	"io/ioutil"
	"os"
	"strings"
	"time"
)

const (
	startDim = "\x1b[2m"
	endDim   = "\x1b[0m"
	startRed = "\x1b[41m"
	endRed   = "\x1b[49m"
)

// Keep in sync with lib/constants.py
const (
	disabledFile   = "/var/run/salt/disabled"
	timeout        = 2 * time.Hour
	serverInfoFile = "/etc/server_info.json"
)

type SwitchPort struct {
	Switch string `json:"switch"`
	Port   string `json:"port"`
}

type ServerInfo struct {
	ID           string       `json:"id"`
	CPUModel     string       `json:"cpu_model"`
	MemTotal     int32        `json:"mem_total"`
	Location     string       `json:"location"`
	Lldp         []SwitchPort `json:"lldp"`
	WalleProject string       `json:"walle_project"`
}

func link(w io.Writer, url, s string) error {
	_, err := fmt.Fprintf(w, "\u001B]8;;%s\u001B\\%s\u001B]8;;\u001B\\", url, s)
	return err
}

func dim(w io.Writer, s string) error {
	_, err := fmt.Fprintf(w, "%s%s%s", startDim, s, endDim)
	return err
}

func red(w io.Writer, s string) error {
	_, err := fmt.Fprintf(w, "%s%s%s", startRed, s, endRed)
	return err
}

func mem(mib int32) string {
	suffix := "MiB"
	tab := []string{"GiB", "TiB"}
	for _, s := range tab {
		if mib < 1024 {
			break
		}
		mib /= 1024
		suffix = s
	}
	return fmt.Sprintf("%d %s", mib, suffix)
}

/* Loads info and formats it to provided writer. Example:
--------
location: sas, switch/port: sas1-s278: GE1/0/10
project: yp-iss-prestable, host: sas2-9707.search.yandex.net ↪ wall-e
hardware: Intel(R) Xeon(R) CPU E5-2660 0 @ 2.20GHz - 62 GiB Mem
--------
*/
func printInfo(w io.Writer, infoPath string) {
	buf, err := ioutil.ReadFile(infoPath)
	if err != nil {
		_, _ = fmt.Fprintf(w, "=== No info: %s ===\n", err)
		return
	}
	info := ServerInfo{}
	if err := json.Unmarshal(buf, &info); err != nil {
		_, _ = fmt.Fprintf(w, "=== No info: %s ===\n", err)
		return
	}
	b := &bytes.Buffer{}
	b.WriteString("--------\n")
	// Location
	_ = dim(b, "location: ")
	if len(info.Location) > 0 {
		b.WriteString(strings.ToLower(info.Location))
	} else {
		b.WriteString("<unknown>")
	}
	// LLDP
	_ = dim(b, ", switch/port: ")
	if len(info.Lldp) > 0 {
		lldp := info.Lldp[0]
		_, _ = fmt.Fprintf(b, "%s: %s", lldp.Switch, lldp.Port)
	}
	b.WriteByte('\n')
	// WALL-E
	_ = dim(b, "project: ")
	if len(info.WalleProject) > 0 {
		b.WriteString(info.WalleProject)
	} else {
		b.WriteString("<unknown>")
	}
	_ = dim(b, ", host: ")
	b.WriteString(info.ID)
	_ = dim(b, " ")
	_ = link(b, fmt.Sprintf("https://wall-e.yandex-team.ru/host/%s", info.ID), "↪ wall-e")
	b.WriteByte('\n')
	// HW
	_ = dim(b, "hardware: ")
	b.WriteString(info.CPUModel)
	b.WriteString(" - ")
	_, _ = fmt.Fprintf(b, "%s Mem\n", mem(info.MemTotal))
	b.WriteString("--------\n")
	_, _ = w.Write(b.Bytes())
}

func printDisabled(w io.Writer) {
	st, err := os.Stat(disabledFile)
	if err != nil {
		// Failed to access file, is it because it does not exist?
		if os.IsNotExist(err) {
			// No file - great!
			return
		}
		if _, err = fmt.Fprintf(w, "%s\n", err.Error()); err != nil {
			panic(err.Error())
		}
		return
	}
	if time.Now().Before(st.ModTime().Add(timeout)) {
		_, _ = io.WriteString(w, "-------\n")
		_ = red(w, "WARNING")
		_, _ = io.WriteString(w, ": ya-salt is disabled (enable via `ya-salt enable`)\n-------\n")
	}
}

func main() {
	printDisabled(os.Stdout)
	// Allow overriding info file for local testing
	infoPath := serverInfoFile
	if len(os.Args) == 2 {
		infoPath = os.Args[1]
	}
	printInfo(os.Stdout, infoPath)
}
