/*
summary of how this script can be called:
       * <postinst> `configure' <most-recently-configured-version>
       * <old-postinst> `abort-upgrade' <new version>
       * <conflictor's-postinst> `abort-remove' `in-favour' <package>
         <new-version>
       * <postinst> `abort-remove'
       * <deconfigured's-postinst> `abort-deconfigure' `in-favour'
         <failed-install-package> <version> `removing'
         <conflicting-package> <version>
for details, see http://www.debian.org/doc/debian-policy/ or the debian-policy package
*/
package main

import (
	"fmt"
	"os"
	"os/exec"
	"path/filepath"
	"syscall"
)

const (
	// Keep in sync with lib/constants.py
	initialSetupFile      = "/var/lib/ya-salt/__need_initial_setup__"
	systemdBin            = "/bin/systemctl"
	bootSystemdSourceFile = "/usr/share/yandex-search-salt/ya-salt-at-boot.service"
	bootSystemdDestFile   = "/etc/systemd/system/ya-salt-at-boot.service"
	initExe               = "/proc/1/exe"
	systemdInitExe        = "/lib/systemd/systemd"
)

var cmdAttr = &syscall.SysProcAttr{
	Pdeathsig: syscall.SIGKILL,
}
var validCommands = []string{
	"abort-upgrade",
	"abort-remove",
	"abort-deconfigure",
}

func makeMarker(path string) error {
	_, err := os.Stat(path)
	if os.IsNotExist(err) {
		dir := filepath.Dir(path)
		err := os.MkdirAll(dir, 0755)
		if err != nil {
			return fmt.Errorf("failed to create '%s': %s", dir, err.Error())
		}
		f, err := os.OpenFile(path, os.O_RDWR|os.O_CREATE|os.O_TRUNC, 0644)
		if err != nil {
			return fmt.Errorf("failed to create '%s': %s", path, err.Error())
		}
		if err := f.Close(); err != nil {
			return fmt.Errorf("failed to close file: %s", err.Error())
		}
	}
	return nil
}

func die(error string) {
	fmt.Println(error)
	os.Exit(1)
}

func contains(haystack []string, needle string) bool {
	for _, s := range haystack {
		if s == needle {
			return true
		}
	}
	return false
}

func installSystemd() error {
	_, err := os.Stat(bootSystemdDestFile)
	if err != nil {
		if os.IsNotExist(err) {
			// This command simply creates symlinks so it can be run
			// without systemd running.
			cmd := exec.Command(systemdBin, "enable", bootSystemdSourceFile)
			cmd.SysProcAttr = cmdAttr
			err := cmd.Run()
			if err != nil {
				return fmt.Errorf("failed to run %+v: %s",
					cmd.Args, err.Error())
			}
			// Need to check if we are in chroot being run not by systemd
			// e.g. during setup.yandex-team.ru
			init, err := os.Readlink(initExe)
			if err != nil {
				return fmt.Errorf("failed to check %s symlink: %s", initExe, err.Error())
			}
			if init != systemdInitExe {
				// Do not call reload, because init is not systemd
				// and DBus call won't succeed.
				return nil
			}
			// Ask systemd to reload it's configuration
			cmd = exec.Command(systemdBin, "daemon-reload")
			cmd.SysProcAttr = cmdAttr
			err = cmd.Run()
			if err != nil {
				return fmt.Errorf("failed to run %+v: %s",
					cmd.Args, err.Error())
			}
		} else {
			return fmt.Errorf("failed to stat %s: %s",
				bootSystemdDestFile, err.Error())
		}
	}
	return nil
}

func configure(prev string) error {
	// No second argument (according to https://wiki.debian.org/MaintainerScripts)
	// means that there was no previous version and this is a fresh installation.
	// However there is caveat, that if someone removes (not purges) a package then
	// it will be left in "Config-Files" state and thus it won't be fresh install,
	// but now it seems okay.
	if len(prev) == 0 {
		err := makeMarker(initialSetupFile)
		if err != nil {
			return err
		}
	}
	// Install at boot service file
	// Cannot check `if systemd` using /run/systemd
	// because during setup there can be no systemd running in chroot
	_, err := os.Stat(systemdBin)
	// Systemd case
	if err == nil {
		return installSystemd()
	}
	// Some other unsupported environment
	return fmt.Errorf("failed to find '%s' - cannot determine if host systemd capable", systemdBin)
}

func main() {
	if len(os.Args) < 2 {
		die("postinst called without arguments")
	}
	command := os.Args[1]
	if command == "configure" {
		prev := ""
		if len(os.Args) == 3 {
			prev = os.Args[2]
		}
		err := configure(prev)
		if err != nil {
			die(err.Error())
		}
	} else {
		if !contains(validCommands, command) {
			die(fmt.Sprint("postinst called with unknown argument ", command))
		}
	}
	os.Exit(0)
}
