"""
This module get info about disks(ssd or non ssd).
"""
import errno
import os

DISK_PREFIXES = (
    'sd',  # SATA/SAS(?) devices
    'nvme',  # NVMe devices
)
MAPPER_PREFIXES = (
    'dm',  # device mapper (LVM) devices
)


def _is_rotational(name):
    """
    Checks if given block device `name` is on rotational media.
    Works both for sd/nvme devices and dm (LVM) devices.
    See: https://github.com/torvalds/linux/blob/master/drivers/md/dm-table.c#L1843
    """
    with open('/sys/block/{}/queue/rotational'.format(name)) as f:
        # Compare full string in case someone will add new values
        # in the future. E.g `10\n`.
        return f.read(128) == '1\n'


def _handle_disk(name, info):
    p = '/dev/{}'.format(name)
    if _is_rotational(name):
        info['nonssd'].append(p)
    else:
        if name.startswith('nvme'):
            info['nvme'].append(p)
        else:
            info['ssd'].append(p)


def _handle_dm(name, info):
    """
    Handles device mapper files (like dm-1).
    They diverge heavily from "conventional" devices as they have logical names
    like "/dev/mapper/ssd-place" which are used for mount.
    I.e. usually we see "/place /dev/mapper/ssd-place" instead of
    "/dev/dm-2".
    So we check storage class for device and add both real path
    and symlink (if exists) to info.
    """
    # Process this disk as conventional one first
    _handle_disk(name, info)
    # Try to read logical name (??) if exists
    try:
        with open('/sys/block/{}/dm/name'.format(name)) as f:
            ln = f.read(128).rstrip()
    except EnvironmentError as e:
        # Okay, no name in sysfs.
        # Not sure if it is okay, but everything is fuzzy here.
        if e.errno == errno.ENOENT:
            return
        raise
    # Check if symlink from logical name exists and add it.
    p = '/dev/mapper/{}'.format(ln)
    if os.path.exists(p):
        if _is_rotational(name):
            info['nonssd'].append(p)
        else:
            info['ssd'].append(p)


def disks_info():
    """
    Returns ssd/non ssd disks.
    """
    info = {'ssd': [], 'nonssd': [], 'nvme': []}
    for name in os.listdir('/sys/block'):
        if name.startswith(DISK_PREFIXES):
            _handle_disk(name, info)
        elif name.startswith(MAPPER_PREFIXES):
            _handle_dm(name, info)
    # Sort output to be able to compare consecutive results,
    # as `listdir` does not guarantee any order.
    info['ssd'].sort()
    info['nonssd'].sort()
    info['nvme'].sort()
    return {'disks_info': info}
