# -*- coding: utf-8 -*-
'''
A module written originally by Armin Ronacher to manage file transfers in an
atomic way
'''

# Import python libs
from __future__ import absolute_import
import os
import tempfile
import shutil

atomic_rename = os.rename  # pylint: disable=C0103
CAN_RENAME_OPEN_FILE = True


class _AtomicWFile(object):
    '''
    Helper class for :func:`atomic_open`.
    '''

    def __init__(self, fhanle, tmp_filename, filename):
        self._fh = fhanle
        self._tmp_filename = tmp_filename
        self._filename = filename

    def __getattr__(self, attr):
        return getattr(self._fh, attr)

    def __enter__(self):
        return self

    def close(self):
        if self._fh.closed:
            return
        self._fh.close()
        if os.path.isfile(self._filename):
            shutil.copymode(self._filename, self._tmp_filename)
        atomic_rename(self._tmp_filename, self._filename)

    def __exit__(self, exc_type, exc_value, traceback):
        if exc_type is None:
            self.close()
        else:
            self._fh.close()
            try:
                os.remove(self._tmp_filename)
            except OSError:
                pass

    def __repr__(self):
        return '<{0} {1}{2}, mode {3}>'.format(
            self.__class__.__name__,
            self._fh.closed and 'closed ' or '',
            self._filename,
            self._fh.mode
        )


def atomic_open(filename, mode='w'):
    '''
    Works like a regular `open()` but writes updates into a temporary
    file instead of the given file and moves it over when the file is
    closed.  The file returned behaves as if it was a regular Python
    '''
    if mode in ('r', 'rb', 'r+', 'rb+', 'a', 'ab'):
        raise TypeError('Read or append modes don\'t work with atomic_open')
    ntf = tempfile.NamedTemporaryFile(mode, prefix='.___atomic_write',
                                      dir=os.path.dirname(filename),
                                      delete=False)
    return _AtomicWFile(ntf, ntf.name, filename)
