# -*- coding: utf-8 -*-
'''
Create and verify ANSI X9.31 RSA signatures using OpenSSL libcrypto
'''

# python libs
from __future__ import absolute_import
# salt libs
import salt.utils
# 3rd-party libs
from cryptography.hazmat.bindings.openssl import binding

_b = binding.Binding()
libcrypto = binding.lib
ffi = binding.ffi

RSA_X931_PADDING = 5


class RSAX931Signer(object):
    '''
    Create ANSI X9.31 RSA signatures using OpenSSL libcrypto
    '''

    def __init__(self, key):
        '''
        Init an RSAX931Signer instance

        :param cryptography.hazmat.backends.openssl.rsa._RSAPrivateKey key: The RSA private key
        '''
        self.key = key

    def sign(self, msg):
        '''
        Sign a message (digest) using the private key

        :param str msg: The message (digest) to sign
        :rtype: str
        :return: The signature, or an empty string if the encryption failed
        '''
        buf = ffi.new("unsigned char[]", libcrypto.RSA_size(self.key._rsa_cdata))
        size = libcrypto.RSA_private_encrypt(len(msg), msg, buf, self.key._rsa_cdata, RSA_X931_PADDING)
        if size < 0:
            raise ValueError('Unable to encrypt message')
        return ffi.unpack(buf, size)


class RSAX931Verifier(object):
    '''
    Verify ANSI X9.31 RSA signatures using OpenSSL libcrypto
    '''

    def __init__(self, key):
        '''
        Init an RSAX931Verifier instance

        :param cryptography.hazmat.backends.openssl.rsa._RSAPrivateKey key: The RSA public key
        '''
        self.key = key

    def verify(self, signed):
        '''
        Recover the message (digest) from the signature using the public key

        :param str signed: The signature created with the private key
        :rtype: str
        :return: The message (digest) recovered from the signature, or an empty
            string if the decryption failed
        '''
        buf = ffi.new("unsigned char[]", libcrypto.RSA_size(self.key._rsa_cdata))
        size = libcrypto.RSA_public_decrypt(len(signed), signed, buf, self.key._rsa_cdata, RSA_X931_PADDING)
        if size < 0:
            code = libcrypto.ERR_get_error()
            buf = ffi.new("char[]", 256)
            libcrypto.ERR_error_string_n(code, buf, len(buf))
            raise ValueError('Unable to decrypt message: {}'.format(ffi.string(buf)))
        return ffi.string(buf)
