#include "handlers.h"

#include <infra/monitoring/common/msgpack.h>
#include <util/string/cast.h>

using namespace NAgent::NPlayer;
using google::protobuf::Arena;
using NYasm::NInterfaces::NInternal::EAgentStatus;

namespace {
    static constexpr TDuration START_TIME_DELAY = TDuration::Seconds(10);
    static constexpr std::array<TStringBuf, 4> AGENT_LOGS = {{
        TStringBuf("core.log"),
        TStringBuf("modules.log"),
        TStringBuf("flood.log"),
        TStringBuf("http.log"),
    }};
}

void TFeedMonitorHandler::DoReply(const NMonitoring::TServiceRequest::TRef request, const TParsedHttpFull&) {
    request->EnableCompression(true);
    THttpResponse response;
    response.SetHttpCode(HttpCodes::HTTP_OK);
    response.SetContentType("text/html");
    response.SetContent(R"(<!DOCTYPE html>
<html>
  <head>
    <title>Feed Monitor</title>
    <link rel="stylesheet" type="text/css" href="https://yasm.yandex-team.ru/feed-monitor/main.css" />
  </head>
  <body>
    <div id="app"></div>
    <script src="https://yasm.yandex-team.ru/feed-monitor/bundle.js"></script>
  </body>
</html>
)");
    request->Finish(response);
}

TBaseHandler::TContext TBaseHandler::CreateContext(const NMonitoring::TServiceRequest::TRef request, const TString& defaultContentType) {
    auto contentType = FindHeader(request->GetInput().Headers(), "Content-Type").GetOrElse(defaultContentType);
    auto acceptType = FindHeader(request->GetInput().Headers(), "Accept").GetOrElse(contentType);
    if (acceptType == TStringBuf("*/*")) {
        acceptType = defaultContentType;
    }

    TParsedHttpFull parsedFirstLine(request->GetInput().FirstLine());
    TString content;
    if (parsedFirstLine.Method == TStringBuf("POST")) {
        content = request->GetInput().ReadAll();
    }

    return TContext{
        .Request=*request,
        .RequestProtocol=GetProtocol(contentType),
        .RequestContent=content,
        .ResponseProtocol=GetProtocol(acceptType)
    };
}

TBaseHandler::TContext TBaseHandler::CreateContext(const NMonitoring::TServiceRequest::TRef request) {
    return CreateContext(request, "application/json");
}

void TBaseHandler::FillResponse(TContext& context) {
    auto acceptEncoding = FindHeader(context.Request.GetInput().Headers(), "Accept-Encoding");
    auto contentEncoding = GetContentEncoding(acceptEncoding);

    context.Response.SetHttpCode(context.ResponceCode);
    context.Response.SetContentType(GetContentType(context.ResponseProtocol));
    if (contentEncoding) {
        context.Request.EnableCompression(true);
        context.Response.AddHeader("Content-Encoding", *contentEncoding);
    }
    context.Response.SetContent(context.ResponseContent);
    context.Request.Finish(context.Response);
}

TBaseHandler::EProtocol TBaseHandler::GetProtocol(const TString& type) {
    if (type.Contains("application/x-protobuf")) {
        return EProtocol::PROTOBUF;
    } else if (type.Contains("application/x-msgpack")) {
        return EProtocol::MSGPACK;
    } else if (type.Contains("application/json")) {
        return EProtocol::JSON;
    } else if (type.Contains("text/plain")) {
        return EProtocol::PLAIN;
    } else {
        return EProtocol::JSON;
    }
}

const TString& TBaseHandler::GetContentType(EProtocol protocol) {
    switch (protocol) {
        case EProtocol::JSON: {
            static const TString json("application/json");
            return json;
        }
        case EProtocol::MSGPACK: {
            static const TString msgpack("application/x-msgpack");
            return msgpack;
        }
        case EProtocol::PROTOBUF: {
            static const TString protobuf("application/x-protobuf");
            return protobuf;
        }
        case EProtocol::PLAIN: {
            static const TString plain("text/plain");
            return plain;
        }
    }
}

TMaybe<TString> TBaseHandler::GetContentEncoding(const TMaybe<TString>& acceptEncoding) {
    if (acceptEncoding.Empty()) {
        return Nothing();
    } else if (acceptEncoding->Contains("*") || acceptEncoding->Contains("gzip")) {
        static const TString gzip("gzip");
        return gzip;
    } else if (acceptEncoding->Contains("z-snappy")) {
        static const TString snappy("z-snappy");
        return snappy;
    } else {
        return Nothing();
    }
}

TMaybe<TString> TBaseHandler::FindHeader(const THttpHeaders& headers, const TStringBuf& name) {
    for (const auto& header : headers) {
        if (AsciiCompareIgnoreCase(header.Name(), name) == 0) {
            return header.Value();
        }
    }
    return Nothing();
}

void TBaseRequestReader::Read(const TBaseHandler::TContext& context) {
    using EProtocol = TBaseHandler::EProtocol;
    switch (context.RequestProtocol) {
        case EProtocol::JSON: {
            NJson::TJsonValue root;
            if (!context.RequestContent.empty()) {
                try {
                    NJson::ReadJsonTree(context.RequestContent, &root, true);
                } catch(const NJson::TJsonException& exc) {
                    ythrow NMonitoring::TBadRequest() << exc.AsStrBuf();
                }
            }
            OnJson(root);
            return;
        }
        case EProtocol::MSGPACK: {
            msgpack::unpacked message;
            if (!context.RequestContent.empty()) {
                try {
                    msgpack::unpack(message, context.RequestContent.data(), context.RequestContent.size());
                } catch(const msgpack::unpack_error& exc) {
                    ythrow NMonitoring::TBadRequest() << exc.what();
                }
            }
            OnMsgPack(message.get());
            return;
        }
        case EProtocol::PROTOBUF: {
            Arena arena;
            auto requestMessage = Arena::CreateMessage<NYasm::NInterfaces::NInternal::TAgentRequest>(&arena);
            if (!requestMessage->ParseFromString(context.RequestContent)) {
                ythrow NMonitoring::TBadRequest() << "Failed to parse protobuf message";
            }
            OnProtobuf(*requestMessage);
            return;
        }
        case EProtocol::PLAIN: {
            OnPlain();
            return;
        }
    }
}

void TBaseRequestReader::OnProtobuf(const NYasm::NInterfaces::NInternal::TAgentRequest&) {
    ythrow NMonitoring::TBadRequest() << "protobuf not supported";
}

void TBaseRequestReader::OnMsgPack(const msgpack::object&) {
    ythrow NMonitoring::TBadRequest() << "msgpack not supported";
}

void TBaseRequestReader::OnJson(const NJson::TJsonValue&) {
    ythrow NMonitoring::TBadRequest() << "json not supported";
}

void TBaseRequestReader::OnPlain() {
}

void TBaseResponseWriter::Write(TBaseHandler::TContext& context) {
    using EProtocol = TBaseHandler::EProtocol;
    switch (context.ResponseProtocol) {
        case EProtocol::JSON: {
            TStringOutput stream(context.ResponseContent);
            NJsonWriter::TBuf buf(NJsonWriter::EHtmlEscapeMode::HEM_UNSAFE, &stream);
            OnJson(buf);
            stream.Finish();
            return;
        }
        case EProtocol::MSGPACK: {
            msgpack::sbuffer buffer;
            msgpack::packer<msgpack::sbuffer> packer(buffer);
            OnMsgPack(packer);
            context.ResponseContent = TStringBuf(buffer.data(), buffer.size());
            return;
        }
        case EProtocol::PROTOBUF: {
            Arena arena;
            auto responseMessage = Arena::CreateMessage<NYasm::NInterfaces::NInternal::TAgentResponse>(&arena);
            OnProtobuf(*responseMessage);
            context.ResponseContent = responseMessage->SerializeAsString();
            return;
        }
        case EProtocol::PLAIN: {
            OnPlain(context.ResponseContent);
            return;
        }
    }
}

void TBaseResponseWriter::OnProtobuf(NYasm::NInterfaces::NInternal::TAgentResponse&) {
    ythrow NMonitoring::TBadRequest() << "protobuf not supported";
}

void TBaseResponseWriter::OnMsgPack(msgpack::packer<msgpack::sbuffer>&) {
    ythrow NMonitoring::TBadRequest() << "msgpack not supported";
}

void TBaseResponseWriter::OnJson(NJsonWriter::TBuf&) {
    ythrow NMonitoring::TBadRequest() << "json not supported";
}

void TBaseResponseWriter::OnPlain(TString&) {
    ythrow NMonitoring::TBadRequest() << "plain text not supported";
}

void TMultipleRequestReader::OnProtobuf(const NYasm::NInterfaces::NInternal::TAgentRequest& request) {
    Handlers.reserve(request.GetHandlers().size());
    for (const auto& handler : request.GetHandlers()) {
        Handlers.emplace_back(handler);
    }
}

void TMultipleRequestReader::OnMsgPack(const msgpack::object& root) {
    if (!root.is_nil()) {
        if (root.type != msgpack::type::MAP) {
            ythrow NMonitoring::TBadRequest() << "not a map given as root";
        }
        for (const auto& pair : NMonitoring::TMapIterator(root.via.map)) {
            Handlers.emplace_back(pair.key.as<TString>());
        }
    }
}

void TMultipleRequestReader::OnJson(const NJson::TJsonValue& root) {
    if (root.IsDefined()) {
        if (!root.IsMap()) {
            ythrow NMonitoring::TBadRequest() << "not a map given as root";
        }
        for (const auto& [name, _] : root.GetMap()) {
            Handlers.emplace_back(name);
        }
    }
}

const TVector<TString>& TMultipleRequestReader::GetHandlers() const {
    return Handlers;
}

void TMultipleResponseWriter::Add(const TString& name, TBaseResponseWriter* writer) {
    Writers.emplace_back(name, writer);
}

void TMultipleResponseWriter::OnProtobuf(NYasm::NInterfaces::NInternal::TAgentResponse& response) {
    for (auto& [_, writer] : Writers) {
        writer->OnProtobuf(response);
    }
}

void TMultipleResponseWriter::OnMsgPack(msgpack::packer<msgpack::sbuffer>& packer) {
    packer.pack_map(Writers.size());
    for (auto& [name, writer] : Writers) {
        NMonitoring::PackString(packer, name);
        writer->OnMsgPack(packer);
    }
}

void TMultipleResponseWriter::OnJson(NJsonWriter::TBuf& buf) {
    buf.BeginObject();
    for (auto& [name, writer] : Writers) {
        buf.WriteKey(name);
        writer->OnJson(buf);
    }
    buf.EndObject();
}

void TPingWriter::OnProtobuf(NYasm::NInterfaces::NInternal::TAgentResponse& response) {
    response.MutablePing()->SetTimestamp(TInstant::Now().Seconds());
}

void TPingWriter::OnMsgPack(msgpack::packer<msgpack::sbuffer>& packer) {
    packer.pack_map(1);
    NMonitoring::PackString(packer, TStringBuf("ping"));
    packer.pack_uint64(TInstant::Now().Seconds());
}

void TPingWriter::OnJson(NJsonWriter::TBuf& buf) {
    buf
        .BeginObject()
        .WriteKey(TStringBuf("ping"))
        .WriteULongLong(TInstant::Now().Seconds())
        .EndObject();
}

void TPingWriter::OnPlain(TString& content) {
    content = ToString(TInstant::Now().Seconds());
}

void TVersionWriter::OnProtobuf(NYasm::NInterfaces::NInternal::TAgentResponse& response) {
    response.MutableVersion()->SetVersion(VersionContainer.Version);
}

void TVersionWriter::OnMsgPack(msgpack::packer<msgpack::sbuffer>& packer) {
    packer.pack_map(1);
    NMonitoring::PackString(packer, TStringBuf("version"));
    NMonitoring::PackString(packer, VersionContainer.Version);
}

void TVersionWriter::OnJson(NJsonWriter::TBuf& buf) {
    buf
        .BeginObject()
        .WriteKey(TStringBuf("version"))
        .WriteString(VersionContainer.Version)
        .EndObject();
}

void TVersionWriter::OnPlain(TString& content) {
    content = VersionContainer.Version;
}

void TAggregatedDataWriter::OnProtobuf(NYasm::NInterfaces::NInternal::TAgentResponse& response) {
    auto playerData(PlayerDataContainer.CheckAndGet());
    TPlayerDataSerializer serializer(*playerData);
    serializer.ToAggregatedProtobuf(*response.MutableAggregatedRecords());
}

void TAggregatedDataWriter::OnMsgPack(msgpack::packer<msgpack::sbuffer>& packer) {
    auto playerData(PlayerDataContainer.CheckAndGet());
    TPlayerDataSerializer serializer(*playerData);
    serializer.ToAggregatedMsgpack(packer);
}

void TAggregatedDataWriter::OnJson(NJsonWriter::TBuf& buf) {
    auto playerData(PlayerDataContainer.CheckAndGet());
    TPlayerDataSerializer serializer(*playerData);
    serializer.ToAggregatedJson(buf);
}

void TPerInstanceDataWriter::OnProtobuf(NYasm::NInterfaces::NInternal::TAgentResponse& response) {
    auto playerData(PlayerDataContainer.CheckAndGet());
    TPlayerDataSerializer serializer(*playerData);
    serializer.ToPerInstanceProtobuf(*response.MutablePerInstanceRecords());
}

void TPerInstanceDataWriter::OnMsgPack(msgpack::packer<msgpack::sbuffer>& packer) {
    auto playerData(PlayerDataContainer.CheckAndGet());
    TPlayerDataSerializer serializer(*playerData);
    serializer.ToPerInstanceMsgpack(packer);
}

void TPerInstanceDataWriter::OnJson(NJsonWriter::TBuf& buf) {
    auto playerData(PlayerDataContainer.CheckAndGet());
    TPlayerDataSerializer serializer(*playerData);
    serializer.ToPerInstanceJson(buf);
}

void TDataStatusWriter::OnProtobuf(NYasm::NInterfaces::NInternal::TAgentResponse& response) {
    response.MutableStatus()->SetStatus(GetStatus());
}

void TDataStatusWriter::OnMsgPack(msgpack::packer<msgpack::sbuffer>& packer) {
    NMonitoring::PackString(packer, GetTextStatus());
}

void TDataStatusWriter::OnJson(NJsonWriter::TBuf& buf) {
    buf.WriteString(GetTextStatus());
}

EAgentStatus TDataStatusWriter::GetStatus() const {
    // TODO: it should use result of previous check-and-get call
    const TInstant now = TInstant::Now();
    auto playerData(PlayerDataContainer.CheckAndGet(now));
    if (now - PlayerDataContainer.GetStartTime() < START_TIME_DELAY) {
        return EAgentStatus::STATUS_STARTING;
    } else if (playerData->Empty()) {
        return EAgentStatus::STATUS_NO_DATA;
    } else {
        return EAgentStatus::STATUS_OK;
    }
}

TStringBuf TDataStatusWriter::GetTextStatus() const {
    switch (GetStatus()) {
        case EAgentStatus::STATUS_STARTING: {
            static constexpr TStringBuf starting = "starting";
            return starting;
        }
        case EAgentStatus::STATUS_NO_DATA: {
            static constexpr TStringBuf noData = "no data";
            return noData;
        }
        case EAgentStatus::STATUS_OK: {
            static constexpr TStringBuf ok = "ok";
            return ok;
        }
        default: {
            static constexpr TStringBuf unknown = "unknown";
            return unknown;
        }
    }
}

void TPingHandler::DoReply(const NMonitoring::TServiceRequest::TRef request, const TParsedHttpFull&) {
    TContext context(CreateContext(request, GetContentType(EProtocol::PLAIN)));
    Writer.Write(context);
    FillResponse(context);
}

void TVersionHandler::DoReply(const NMonitoring::TServiceRequest::TRef request, const TParsedHttpFull&) {
    TContext context(CreateContext(request, GetContentType(EProtocol::PLAIN)));
    Writer.Write(context);
    FillResponse(context);
}

void TLogHandler::DoReply(const NMonitoring::TServiceRequest::TRef request, const TParsedHttpFull&) {
    request->EnableCompression(true);
    THttpResponse response;
    response.SetHttpCode(HttpCodes::HTTP_OK);
    response.SetContentType("text/plain");
    response.SetContent(TFileInput(LogPath).ReadAll());
    request->Finish(response);
}

void TPerInstanceDataHandler::DoReply(const NMonitoring::TServiceRequest::TRef request, const TParsedHttpFull&) {
    TContext context(CreateContext(request, GetContentType(EProtocol::JSON)));
    MultipleWriter.Write(context);
    FillResponse(context);
}

void TAggregatedDataHandler::DoReply(const NMonitoring::TServiceRequest::TRef request, const TParsedHttpFull&) {
    TContext context(CreateContext(request, GetContentType(EProtocol::JSON)));
    MultipleWriter.Write(context);
    FillResponse(context);
}

void TMultiHandler::DoReply(const NMonitoring::TServiceRequest::TRef request, const TParsedHttpFull&) {
    TContext context(CreateContext(request, GetContentType(EProtocol::JSON)));

    TMultipleResponseWriter writer;
    writer.Add("status", &StatusWriter);

    TMultipleRequestReader reader;
    reader.Read(context);

    bool found = false;
    for (const auto& name : reader.GetHandlers()) {
        const auto it(WriterMap.find(name));
        if (it != WriterMap.end() && it->first != TStringBuf("status")) {
            writer.Add(name, it->second);
            found = true;
        }
    }

    if (!found) {
        writer.Add("get", &PerInstanceWriter);
    }

    writer.Write(context);
    FillResponse(context);
}

THandlersCollection::THandlersCollection(TLog& logger, const TPlayerDataContainer& playerDataContainer, const TVersionContainer& versionContainer, const TString& logDir) {
    // Debug handle.  May be used to check agent availability.
    Handlers.emplace_back("/ping", MakeHolder<TPingHandler>());
    Handlers.emplace_back("/ping/", MakeHolder<TPingHandler>());

    // Version handle returns the current agent version. May be used to check agent availability.
    Handlers.emplace_back("/version", MakeHolder<TVersionHandler>(versionContainer));
    Handlers.emplace_back("/version/", MakeHolder<TVersionHandler>(versionContainer));

    // Json handle is a multi handler that can return multiple types of information. See implementation for details.
    // Handle is called by yasmserver's requesters in protobuf format.
    Handlers.emplace_back("/json", MakeHolder<TMultiHandler>(playerDataContainer, versionContainer));
    Handlers.emplace_back("/json/", MakeHolder<TMultiHandler>(playerDataContainer, versionContainer));

    // Debug handle. Return per-instance data.
    Handlers.emplace_back("/json/get", MakeHolder<TPerInstanceDataHandler>(playerDataContainer));
    Handlers.emplace_back("/json/get/", MakeHolder<TPerInstanceDataHandler>(playerDataContainer));

    // Debug handle. Return aggregated data.
    Handlers.emplace_back("/json/aggr", MakeHolder<TAggregatedDataHandler>(playerDataContainer));
    Handlers.emplace_back("/json/aggr/", MakeHolder<TAggregatedDataHandler>(playerDataContainer));

    // Serves feed-monitor app used by users debugging their signals on host level.
    Handlers.emplace_back("/feed-monitor", MakeHolder<TFeedMonitorHandler>());
    Handlers.emplace_back("/feed-monitor/", MakeHolder<TFeedMonitorHandler>());

    // Multi handler that can return multiple types of information. See implementation for details.
    // Called by dom0 agent on subagents running inside containers.
    Handlers.emplace_back("/multi", MakeHolder<TMultiHandler>(playerDataContainer, versionContainer));
    Handlers.emplace_back("/multi/", MakeHolder<TMultiHandler>(playerDataContainer, versionContainer));

    // Debug handles used to extract yasmagent logs. Useful for fast retrieval of logs from hosts with no ssh access.
    for (const auto& agentLog: AGENT_LOGS) {
        Handlers.emplace_back("/logs/" + ToString(agentLog), MakeHolder<TLogHandler>(logDir + '/' + agentLog));
    }

    Y_UNUSED(logger);
}

void THandlersCollection::Register(NMonitoring::TWebServer& server) {
    for (const auto& pair : Handlers) {
        server.Add(pair.first, *pair.second);
    }
}

TPlayer::TPlayer(const THttpServerOptions& serverOptions, const TString& version, const TString& logDir)
    : VersionContainer{.Version = version}
    , Handlers(Logger, PlayerDataContainer, VersionContainer, logDir)
    , HttpServer(serverOptions)
{
}

void TPlayer::SetPlayerData(TPlayerData* playerData) noexcept {
    PlayerDataContainer.Set(TAtomicSharedPtr<TPlayerData>(playerData));
}

void TPlayer::Run() {
    Handlers.Register(HttpServer);
    HttpServer.StartServing();
    Event.Wait();
}

void TPlayer::Stop() {
    Event.Signal();
    HttpServer.Stop();
}
