#pragma once

#include "types.h"
#include "containers.h"
#include "serializers.h"

#include <infra/monitoring/common/web_handlers.h>
#include <infra/monitoring/common/web_server.h>
#include <infra/yasm/interfaces/internal/agent.pb.h>

#include <contrib/libs/msgpack/include/msgpack.hpp>
#include <util/stream/file.h>

namespace NAgent::NPlayer {
    class TFeedMonitorHandler final : public NMonitoring::IServiceReplier {
    public:
        void DoReply(const NMonitoring::TServiceRequest::TRef request, const TParsedHttpFull& meta) override;
    };

    class TBaseHandler : public NMonitoring::IServiceReplier {
    public:
        virtual ~TBaseHandler() = default;

        enum class EProtocol {
            JSON,
            MSGPACK,
            PROTOBUF,
            PLAIN,
        };

        struct TContext {
            const NMonitoring::TServiceRequest& Request;
            EProtocol RequestProtocol = EProtocol::JSON;
            TString RequestContent;

            THttpResponse Response;
            EProtocol ResponseProtocol = EProtocol::JSON;
            TString ResponseContent;
            HttpCodes ResponceCode = HttpCodes::HTTP_OK;
        };

    protected:
        static TContext CreateContext(const NMonitoring::TServiceRequest::TRef request, const TString& defaultContentType);
        static TContext CreateContext(const NMonitoring::TServiceRequest::TRef request);
        static void FillResponse(TContext& context);

        static EProtocol GetProtocol(const TString& type);
        static const TString& GetContentType(EProtocol protocol);
        static TMaybe<TString> GetContentEncoding(const TMaybe<TString>& acceptEncoding);

        static TMaybe<TString> FindHeader(const THttpHeaders& headers, const TStringBuf& name);
    };

    class TBaseRequestReader : public TMoveOnly {
    public:
        virtual ~TBaseRequestReader() = default;

        void Read(const TBaseHandler::TContext& context);

        virtual void OnProtobuf(const NYasm::NInterfaces::NInternal::TAgentRequest& request);
        virtual void OnMsgPack(const msgpack::object& root);
        virtual void OnJson(const NJson::TJsonValue& root);
        virtual void OnPlain();
    };

    class TBaseResponseWriter : public TMoveOnly {
    public:
        virtual ~TBaseResponseWriter() = default;

        void Write(TBaseHandler::TContext& context);

        virtual void OnProtobuf(NYasm::NInterfaces::NInternal::TAgentResponse& response);
        virtual void OnMsgPack(msgpack::packer<msgpack::sbuffer>& packer);
        virtual void OnJson(NJsonWriter::TBuf& buf);
        virtual void OnPlain(TString& content);
    };

    class TMultipleRequestReader final : public TBaseRequestReader {
    public:
        void OnProtobuf(const NYasm::NInterfaces::NInternal::TAgentRequest& request) override;
        void OnMsgPack(const msgpack::object& root) override;
        void OnJson(const NJson::TJsonValue& root) override;

        const TVector<TString>& GetHandlers() const;

    private:
        TVector<TString> Handlers;
    };

    class TMultipleResponseWriter final : public TBaseResponseWriter {
    public:
        void Add(const TString& name, TBaseResponseWriter* writer);

        void OnProtobuf(NYasm::NInterfaces::NInternal::TAgentResponse& response) override;
        void OnMsgPack(msgpack::packer<msgpack::sbuffer>& packer) override;
        void OnJson(NJsonWriter::TBuf& buf) override;

    private:
        TVector<std::pair<TString, TBaseResponseWriter*>> Writers;
    };

    class TPingWriter final : public TBaseResponseWriter {
    public:
        void OnProtobuf(NYasm::NInterfaces::NInternal::TAgentResponse& response) override;
        void OnMsgPack(msgpack::packer<msgpack::sbuffer>& packer) override;
        void OnJson(NJsonWriter::TBuf& buf) override;
        void OnPlain(TString& content) override;
    };

    class TVersionWriter final : public TBaseResponseWriter {
    public:
        TVersionWriter(const TVersionContainer& versionContainer)
            : VersionContainer(versionContainer)
        {
        }

        void OnProtobuf(NYasm::NInterfaces::NInternal::TAgentResponse& response) override;
        void OnMsgPack(msgpack::packer<msgpack::sbuffer>& packer) override;
        void OnJson(NJsonWriter::TBuf& buf) override;
        void OnPlain(TString& content) override;

    private:
        const TVersionContainer& VersionContainer;
    };

    class TAggregatedDataWriter final : public TBaseResponseWriter {
    public:
        TAggregatedDataWriter(const TPlayerDataContainer& playerDataContainer)
            : PlayerDataContainer(playerDataContainer)
        {
        }

        void OnProtobuf(NYasm::NInterfaces::NInternal::TAgentResponse& response) override;
        void OnMsgPack(msgpack::packer<msgpack::sbuffer>& packer) override;
        void OnJson(NJsonWriter::TBuf& buf) override;

    private:
        const TPlayerDataContainer& PlayerDataContainer;
    };

    class TPerInstanceDataWriter final : public TBaseResponseWriter {
    public:
        TPerInstanceDataWriter(const TPlayerDataContainer& playerDataContainer)
            : PlayerDataContainer(playerDataContainer)
        {
        }

        void OnProtobuf(NYasm::NInterfaces::NInternal::TAgentResponse& response) override;
        void OnMsgPack(msgpack::packer<msgpack::sbuffer>& packer) override;
        void OnJson(NJsonWriter::TBuf& buf) override;

    private:
        const TPlayerDataContainer& PlayerDataContainer;
    };

    class TDataStatusWriter final : public TBaseResponseWriter {
    public:
        TDataStatusWriter(const TPlayerDataContainer& playerDataContainer)
            : PlayerDataContainer(playerDataContainer)
        {
        }

        void OnProtobuf(NYasm::NInterfaces::NInternal::TAgentResponse& response) override;
        void OnMsgPack(msgpack::packer<msgpack::sbuffer>& packer) override;
        void OnJson(NJsonWriter::TBuf& buf) override;

    private:
        NYasm::NInterfaces::NInternal::EAgentStatus GetStatus() const;
        TStringBuf GetTextStatus() const;

        const TPlayerDataContainer& PlayerDataContainer;
    };

    class TPingHandler final : public TBaseHandler {
    public:
        void DoReply(const NMonitoring::TServiceRequest::TRef request, const TParsedHttpFull& meta) override;

    private:
        TPingWriter Writer;
    };

    class TVersionHandler final : public TBaseHandler {
    public:
        TVersionHandler(const TVersionContainer& versionContainer)
            : Writer(versionContainer)
        {
        }

        void DoReply(const NMonitoring::TServiceRequest::TRef request, const TParsedHttpFull& meta) override;

    private:
        TVersionWriter Writer;
    };

    class TLogHandler final: public TBaseHandler {
    public:
        TLogHandler(const TString& logPath)
            : LogPath(logPath)
        {
        }

        void DoReply(const NMonitoring::TServiceRequest::TRef request, const TParsedHttpFull& meta) override;
    private:
        const TString LogPath;
    };

    class TPerInstanceDataHandler final : public TBaseHandler {
    public:
        TPerInstanceDataHandler(const TPlayerDataContainer& playerDataContainer)
            : DataWriter(playerDataContainer)
            , StatusWriter(playerDataContainer)
        {
            MultipleWriter.Add("status", &StatusWriter);
            MultipleWriter.Add("get", &DataWriter);
        }

        void DoReply(const NMonitoring::TServiceRequest::TRef request, const TParsedHttpFull& meta) override;

    private:
        TPerInstanceDataWriter DataWriter;
        TDataStatusWriter StatusWriter;
        TMultipleResponseWriter MultipleWriter;
    };

    class TAggregatedDataHandler final : public TBaseHandler {
    public:
        TAggregatedDataHandler(const TPlayerDataContainer& playerDataContainer)
            : DataWriter(playerDataContainer)
            , StatusWriter(playerDataContainer)
        {
            MultipleWriter.Add("status", &StatusWriter);
            MultipleWriter.Add("aggr", &DataWriter);
        }

        void DoReply(const NMonitoring::TServiceRequest::TRef request, const TParsedHttpFull& meta) override;

    private:
        TAggregatedDataWriter DataWriter;
        TDataStatusWriter StatusWriter;
        TMultipleResponseWriter MultipleWriter;
    }
    ;
    class TMultiHandler final : public TBaseHandler {
    public:
        TMultiHandler(const TPlayerDataContainer& playerDataContainer, const TVersionContainer& versionContainer)
            : AggregatedWriter(playerDataContainer)
            , PerInstanceWriter(playerDataContainer)
            , StatusWriter(playerDataContainer)
            , VersionWriter(versionContainer)
        {
            WriterMap.emplace("aggr", &AggregatedWriter);
            WriterMap.emplace("get", &PerInstanceWriter);
            WriterMap.emplace("status", &StatusWriter);
            WriterMap.emplace("version", &VersionWriter);
            WriterMap.emplace("ping", &PingWriter);
        }

        void DoReply(const NMonitoring::TServiceRequest::TRef request, const TParsedHttpFull& meta) override;

    private:
        TAggregatedDataWriter AggregatedWriter;
        TPerInstanceDataWriter PerInstanceWriter;
        TDataStatusWriter StatusWriter;
        TVersionWriter VersionWriter;
        TPingWriter PingWriter;
        THashMap<TString, TBaseResponseWriter*> WriterMap;
    };

    class THandlersCollection {
    public:
        THandlersCollection(TLog& logger, const TPlayerDataContainer& playerDataContainer, const TVersionContainer& versionContainer, const TString& logDir);

        void Register(NMonitoring::TWebServer& server);

    private:
        TVector<std::pair<TString, THolder<NMonitoring::IServiceReplier>>> Handlers;
    };

    class TPlayer {
    public:
        TPlayer(const THttpServerOptions& serverOptions, const TString& version, const TString& logDir);

        void SetPlayerData(TPlayerData* data) noexcept;

        void Run();
        void Stop();

    private:
        TLog Logger;

        TPlayerDataContainer PlayerDataContainer;
        TVersionContainer VersionContainer;

        THandlersCollection Handlers;
        NMonitoring::TWebServer HttpServer;

        TAutoEvent Event;
    };
}
