#include "parser.h"

#include <infra/yasm/zoom/components/serialization/python/py_to_zoom.h>

#include <library/cpp/pybind/ptr.h>

#include <util/generic/xrange.h>

using namespace NAgent::NPlayer;
using namespace NZoom::NRecord;
using namespace NZoom::NPython;

namespace {
    TString PyObjectToString(PyObject* obj) {
        char* buf;
        Py_ssize_t len;
        if (PyUnicode_Check(obj)) {
            NPyBind::TPyObjectPtr encbytes(PyUnicode_AsASCIIString(obj), true);
            return TString(PyBytes_AsString(encbytes.Get()), PyBytes_Size(encbytes.Get()));
        } else if (!PyString_Check(obj) || PyString_AsStringAndSize(obj, &buf, &len)) {
            ythrow yexception() << "obj is not string";
        }
        return TString(buf, len);
    }

    template <class F>
    void IteratePyDict(PyObject* root, F cb) {
        PyObject *itemKey, *itemValue;
        Py_ssize_t pos = 0;
        while (PyDict_Next(root, &pos, &itemKey, &itemValue)) {
            cb(PyObjectToString(itemKey), itemValue);
        }
    }

    template <class F>
    void IteratePyList(PyObject* root, F cb) {
        for (const auto position : xrange(PyList_Size(root))) {
            cb(PyList_GET_ITEM(root, position));
        }
    }
}

TPlayerData* TPlayerDataParser::Deserialize(TInstant timestamp, PyObject* perInstanceData, PyObject* aggregatedData) {
    THolder<TPlayerData> playerData(MakeHolder<TPlayerData>(
        timestamp, DeserializePerInstanceDataList(perInstanceData), DeserializeAggregatedDataList(aggregatedData)
    ));
    return playerData.Release();
}

TVector<NTags::TInstanceKey> TPlayerDataParser::DeserializeTags(PyObject* root) {
    if (!PyList_Check(root)) {
        ythrow yexception() << "per_instance_data is not list";
    }

    TVector<NTags::TInstanceKey> tags(Reserve(PyList_Size(root)));
    IteratePyList(root, [&](PyObject* tag) {
        tags.emplace_back(NTags::TInstanceKey::FromNamed(PyObjectToString(tag)));
    });

    return tags;
}

TPerInstanceData TPlayerDataParser::DeserializePerInstanceData(const TString& instanceName, PyObject* data) {
    if (!PyDict_Check(data)) {
        ythrow yexception() << "per_instance_data is not dict";
    }

    TString hostName;
    TString instanceType;
    TVector<NTags::TInstanceKey> tags;
    THolder<TRecord> record;
    IteratePyDict(data, [&](TString key, PyObject* value) {
        if (key == TStringBuf("hostname")) {
            hostName = PyObjectToString(value);
        } else if (key == TStringBuf("tags")) {
            tags = DeserializeTags(value);
        } else if (key == TStringBuf("type")) {
            instanceType = PyObjectToString(value);
        } else if (key == TStringBuf("values")) {
            record.Reset(TPyDeserializer::DeserializeDict(value, false));
        } else {
            ythrow yexception() << "wrong key " << key << " found in per_instance_data";
        }
    });

    return TPerInstanceData(instanceName, instanceType, hostName, std::move(tags), std::move(*record));
}

TAggregatedData TPlayerDataParser::DeserializeAggregatedData(PyObject* data) {
    if (!PyTuple_Check(data)) {
        ythrow yexception() << "aggregated_data is not tuple";
    }
    if (PyTuple_GET_SIZE(data) != 3) {
        ythrow yexception() << "aggregated_data size isn't 3";
    }

    TString instanceType = PyObjectToString(PyTuple_GET_ITEM(data, 0));
    TString instanceTail = PyObjectToString(PyTuple_GET_ITEM(data, 1));
    THolder<TRecord> record(TPyDeserializer::DeserializeDict(PyTuple_GET_ITEM(data, 2), false));

    return TAggregatedData(instanceType, instanceTail, std::move(*record));
}

TVector<TPerInstanceData> TPlayerDataParser::DeserializePerInstanceDataList(PyObject* root) {
    if (!PyDict_Check(root)) {
        ythrow yexception() << "aggregated_data is not dict";
    }

    TVector<TPerInstanceData> elements(Reserve(PyDict_Size(root)));
    IteratePyDict(root, [&](TString key, PyObject* data) {
        elements.emplace_back(DeserializePerInstanceData(key, data));
    });

    return elements;
}

TVector<TAggregatedData> TPlayerDataParser::DeserializeAggregatedDataList(PyObject* root) {
    if (!PyList_Check(root)) {
        ythrow yexception() << "per_instance_data is not list";
    }

    TVector<TAggregatedData> elements(Reserve(PyList_Size(root)));
    IteratePyList(root, [&](PyObject* data) {
        elements.emplace_back(DeserializeAggregatedData(data));
    });

    return elements;
}
