#include "serializers.h"

#include <infra/yasm/common/points/hgram/ugram/compress/compress.h>
#include <infra/yasm/zoom/components/serialization/zoom_to_protobuf/instance_key.h>
#include <infra/yasm/zoom/components/serialization/zoom_to_protobuf/record.h>

#include <infra/monitoring/common/msgpack.h>

using namespace NAgent::NPlayer;
using namespace NZoom::NProtobuf;

void TPlayerDataSerializer::ToPerInstanceJson(NJsonWriter::TBuf& jsonBuf) {
    jsonBuf.BeginObject();
    for (const auto& perInstanceData : PlayerData.GetPerInstance()) {
        jsonBuf.WriteKey(perInstanceData.GetName());

        jsonBuf
            .BeginObject()
            .WriteKey(TStringBuf("hostname"))
            .WriteString(perInstanceData.GetHostName())
            .WriteKey(TStringBuf("type"))
            .WriteString(perInstanceData.GetItype());

        jsonBuf
            .WriteKey(TStringBuf("tags"))
            .BeginList();
        for (const auto& tag : perInstanceData.GetTags()) {
            jsonBuf.WriteString(tag.ToNamed());
        }
        jsonBuf.EndList();

        jsonBuf.WriteKey(TStringBuf("values"));
        NZoom::NPython::TJsonSerializer<NZoom::NHgram::TLazyUgramCompressor> serializer(jsonBuf);
        perInstanceData.GetRecord().Process(serializer);

        jsonBuf.EndObject();
    }
    jsonBuf.EndObject();
}

TString TPlayerDataSerializer::ToPerInstanceJson(const TPlayerData& playerData) {
    NJsonWriter::TBuf jsonBuf;
    TPlayerDataSerializer serializer(playerData);
    serializer.ToPerInstanceJson(jsonBuf);
    return jsonBuf.Str();
}

void TPlayerDataSerializer::ToAggregatedJson(NJsonWriter::TBuf& jsonBuf) {
    jsonBuf.BeginList();
    for (const auto& aggregatedData : PlayerData.GetAggregated()) {
        jsonBuf
            .BeginList()
            .WriteString(aggregatedData.GetItype())
            .WriteString(aggregatedData.GetTail());

        NZoom::NPython::TJsonSerializer<NZoom::NHgram::TLazyUgramCompressor> serializer(jsonBuf);
        aggregatedData.GetRecord().Process(serializer);

        jsonBuf.EndList();
    }
    jsonBuf.EndList();
}

TString TPlayerDataSerializer::ToAggregatedJson(const TPlayerData& playerData) {
    NJsonWriter::TBuf jsonBuf;
    TPlayerDataSerializer serializer(playerData);
    serializer.ToAggregatedJson(jsonBuf);
    return jsonBuf.Str();
}

void TPlayerDataSerializer::ToPerInstanceMsgpack(msgpack::packer<msgpack::sbuffer>& packer) {
    packer.pack_map(PlayerData.GetPerInstance().size());
    for (const auto& perInstanceData : PlayerData.GetPerInstance()) {
        NMonitoring::PackString(packer, perInstanceData.GetName());
        packer.pack_map(4);

        NMonitoring::PackString(packer, TStringBuf("hostname"));
        NMonitoring::PackString(packer, perInstanceData.GetHostName());

        NMonitoring::PackString(packer, TStringBuf("type"));
        NMonitoring::PackString(packer, perInstanceData.GetItype());

        NMonitoring::PackString(packer, TStringBuf("tags"));
        packer.pack_array(perInstanceData.GetTags().size());
        for (const auto& tag : perInstanceData.GetTags()) {
            NMonitoring::PackString(packer, tag.ToNamed());
        }

        NMonitoring::PackString(packer, TStringBuf("values"));
        NZoom::NPython::TMsgPackRefSerializer serializer(packer, false);
        perInstanceData.GetRecord().Process(serializer);
    }
}

TString TPlayerDataSerializer::ToPerInstanceMsgpack(const TPlayerData& playerData) {
    msgpack::sbuffer buf;
    msgpack::packer<msgpack::sbuffer> packer(buf);
    TPlayerDataSerializer serializer(playerData);
    serializer.ToPerInstanceMsgpack(packer);
    return TString(buf.data(), buf.size());
}

void TPlayerDataSerializer::ToAggregatedMsgpack(msgpack::packer<msgpack::sbuffer>& packer) {
    packer.pack_array(PlayerData.GetAggregated().size());
    for (const auto& aggregatedData : PlayerData.GetAggregated()) {
        packer.pack_array(3);
        NMonitoring::PackString(packer, aggregatedData.GetItype());
        NMonitoring::PackString(packer, aggregatedData.GetTail());
        NZoom::NPython::TMsgPackRefSerializer serializer(packer, false);
        aggregatedData.GetRecord().Process(serializer);
    }
}

TString TPlayerDataSerializer::ToAggregatedMsgpack(const TPlayerData& playerData) {
    msgpack::sbuffer buf;
    msgpack::packer<msgpack::sbuffer> packer(buf);
    TPlayerDataSerializer serializer(playerData);
    serializer.ToAggregatedMsgpack(packer);
    return TString(buf.data(), buf.size());
}

void TPlayerDataSerializer::ToPerInstanceProtobuf(NYasm::NInterfaces::NInternal::TPerInstanceRecords& records) {
    TSignalNameSerializer signalNameSerializer(records.MutableSignalNameTable());
    TInstanceKeySerializer instanceKeySerializer(records.MutableInstanceKeyTable());
    records.MutableRecords()->Reserve(PlayerData.GetPerInstance().size());
    for (const auto& perInstanceData : PlayerData.GetPerInstance()) {
        auto& row(*records.MutableRecords()->Add());
        row.SetInstanceName(perInstanceData.GetName());
        row.SetInstanceType(perInstanceData.GetItype());
        row.SetHostName(perInstanceData.GetHostName());

        row.MutableInstanceKeys()->Reserve(perInstanceData.GetTags().size());
        for (const auto& instanceKey : perInstanceData.GetTags()) {
            instanceKeySerializer.Intern(instanceKey, row.MutableInstanceKeys()->Add());
        }

        TProtobufRecordSerializer<NZoom::NHgram::TLazyUgramCompressor> serializer(*row.MutableRecord(), signalNameSerializer);
        perInstanceData.GetRecord().Process(serializer);
    }
}

TString TPlayerDataSerializer::ToPerInstanceProtobuf(const TPlayerData& playerData) {
    google::protobuf::Arena arena;
    auto message = google::protobuf::Arena::CreateMessage<NYasm::NInterfaces::NInternal::TPerInstanceRecords>(&arena);
    TPlayerDataSerializer serializer(playerData);
    serializer.ToPerInstanceProtobuf(*message);
    return message->SerializeAsString();
}

void TPlayerDataSerializer::ToAggregatedProtobuf(NYasm::NInterfaces::NInternal::TAggregatedRecords& records) {
    TSignalNameSerializer signalNameSerializer(records.MutableSignalNameTable());
    TInstanceKeySerializer instanceKeySerializer(records.MutableInstanceKeyTable());
    records.MutableRecords()->Reserve(PlayerData.GetAggregated().size());
    for (const auto& aggregatedData : PlayerData.GetAggregated()) {
        auto& row(*records.MutableRecords()->Add());
        instanceKeySerializer.Intern(aggregatedData.GetInstanceKey(), row.MutableInstanceKey());

        TProtobufRecordSerializer<NZoom::NHgram::TLazyUgramCompressor> serializer(*row.MutableRecord(), signalNameSerializer);
        aggregatedData.GetRecord().Process(serializer);
    }
}

TString TPlayerDataSerializer::ToAggregatedProtobuf(const TPlayerData& playerData) {
    google::protobuf::Arena arena;
    auto message = google::protobuf::Arena::CreateMessage<NYasm::NInterfaces::NInternal::TAggregatedRecords>(&arena);
    TPlayerDataSerializer serializer(playerData);
    serializer.ToAggregatedProtobuf(*message);
    return message->SerializeAsString();
}
