#include "ast.h"
#include "exceptions.h"

#include <util/stream/output.h>
#include <util/stream/str.h>
#include <util/string/cast.h>

using namespace NYasm::NAldan;

namespace {
    class TPrintingNodeVisitor final : public TNodeVisitor {
    public:
        enum TActionType {
            NOTHING,
            FIRST_ARGUMENT,
            NEXT_ARGUMENT
        };

        TPrintingNodeVisitor(IOutputStream& stream)
            : Stream(stream)
            , ActionStack({NOTHING})
        {
        }

        void EnterNode(TNode&) override {
            Y_FAIL();
        }

        void EnterNode(const TNode&) override {
            switch (ActionStack.back()) {
                case NOTHING: {
                    break;
                }
                case FIRST_ARGUMENT: {
                    ActionStack.back() = NEXT_ARGUMENT;
                    break;
                }
                case NEXT_ARGUMENT: {
                    Stream.Write(TStringBuf(", "));
                    break;
                }
            }
            ActionStack.emplace_back(NOTHING);
        }

        void VisitInteger(i64 number) override {
            Stream.Write(ToString(number));
        }

        void VisitDouble(double number) override {
            Stream.Write(ToString(number));
        }

        void VisitIdent(const TString& ident) override {
            Stream.Write(ident);
        }

        void VisitApply(const TString& name) override {
            Stream.Write(name);
            Stream.Write('(');
            ActionStack.back() = FIRST_ARGUMENT;
        }

        void FinishNode() override {
            switch (ActionStack.back()) {
                case NOTHING: {
                    break;
                }
                case FIRST_ARGUMENT:
                case NEXT_ARGUMENT: {
                    Stream.Write(')');
                    break;
                }
            }
            ActionStack.pop_back();
        }

    private:
        IOutputStream& Stream;
        TVector<TActionType> ActionStack;
    };
}

void TNode::AddChild(THolder<TNode>) {
    ythrow TParsingError() << "child can't be added to this node";
}

TTypeConstructor::TPtr TNode::GetInferredType() const {
    if (!Definition) {
        ythrow TInferError() << "node type can't be detected";
    }
    return Definition->GetInferredType();
}

void TNode::SetDefinition(TDefinition::TPtr definition) {
    Definition = std::move(definition);
}

TString TNode::ToString() const {
    TString result;
    TStringOutput stream(result);
    TPrintingNodeVisitor visitor(stream);
    Visit(visitor);
    stream.Finish();
    return result;
}

template <>
void Out<TNode>(IOutputStream& stream,
                TTypeTraits<TNode>::TFuncParam node) {
    TPrintingNodeVisitor visitor(stream);
    node.Visit(visitor);
}
