#include "augment.h"
#include "builtin_types.h"

#include <infra/yasm/aldan/parser/parser.h>

#include <library/cpp/testing/unittest/registar.h>

using namespace NYasm::NAldan;

namespace {
    class TCollectingVisitor final : public TNodeVisitor {
    public:
        void EnterNode(const TNode& node) override {
            Types.emplace_back(node.GetInferredType());
        }

        void VisitInteger(i64 integer) override {
            Names.emplace_back(ToString(integer));
        }

        void VisitDouble(double number) override {
            Names.emplace_back(ToString(number));
        }

        void VisitIdent(const TString& ident) override {
            Names.emplace_back(ident);
        }

        void VisitApply(const TString& name) override {
            Names.emplace_back(name);
        }

        TVector<std::pair<TString, TTypeConstructor::TPtr>> GetTypes() {
            TVector<std::pair<TString, TTypeConstructor::TPtr>> result(Reserve(Types.size()));
            UNIT_ASSERT_VALUES_EQUAL(Types.size(), Names.size());
            for (const auto index : xrange(Types.size())) {
                result.emplace_back(std::move(Names[index]), std::move(Types[index]));
            }
            return result;
        }

    private:
        TVector<TTypeConstructor::TPtr> Types;
        TVector<TString> Names;
    };
}

Y_UNIT_TEST_SUITE(TAldanAugmentTest) {

    Y_UNIT_TEST(TestSomething) {
        TRegistry registry;
        registry["GiB"].emplace_back(DOUBLE);
        registry["hmerge"].emplace_back(MakeIntrusive<TFunctionConstructor>(
            MakeIntrusive<TListConstructor>(HISTOGRAM_VALUE),
            HISTOGRAM_VALUE
        ));
        registry["havg"].emplace_back(MakeIntrusive<TFunctionConstructor>(
            MakeIntrusive<TTupleConstructor>(TVector<TTypeConstructor::TPtr>{
                HISTOGRAM_VALUE
            }),
            DOUBLE_VALUE
        ));
        registry["conv"].emplace_back(MakeIntrusive<TFunctionConstructor>(
            MakeIntrusive<TTupleConstructor>(TVector<TTypeConstructor::TPtr>{
                DOUBLE_VALUE,
                DOUBLE
            }),
            DOUBLE_VALUE
        ));
        registry["plus"].emplace_back(MakeIntrusive<TFunctionConstructor>(
            MakeIntrusive<TTupleConstructor>(TVector<TTypeConstructor::TPtr>{
                DOUBLE_VALUE,
                DOUBLE
            }),
            DOUBLE_VALUE
        ));

        auto rootNode = ParseExpression("conv(havg(hmerge(first_ahhh, second_ahhh)), GiB) + 5.0");
        UNIT_ASSERT(rootNode);

        auto resultType = AugmentAST(*rootNode, registry);
        UNIT_ASSERT_VALUES_EQUAL(*resultType, *DOUBLE_VALUE);

        TCollectingVisitor visitor;
        rootNode->Visit(visitor);
        auto inferredTypes(visitor.GetTypes());

        TVector<std::pair<TString, TTypeConstructor::TPtr>> expectedTypes;
        expectedTypes.emplace_back("plus", registry["plus"].back());
        expectedTypes.emplace_back("conv", registry["conv"].back());
        expectedTypes.emplace_back("havg", registry["havg"].back());
        expectedTypes.emplace_back("hmerge", registry["hmerge"].back());
        expectedTypes.emplace_back("first_ahhh", HISTOGRAM_VALUE);
        expectedTypes.emplace_back("second_ahhh", HISTOGRAM_VALUE);
        expectedTypes.emplace_back("GiB", DOUBLE);
        expectedTypes.emplace_back("5", DOUBLE);
        UNIT_ASSERT_VALUES_EQUAL(inferredTypes.size(), expectedTypes.size());
        for (const auto index : xrange(expectedTypes.size())) {
            auto& left(inferredTypes[index]);
            auto& right(expectedTypes[index]);
            UNIT_ASSERT_VALUES_EQUAL(left.first, right.first);
            UNIT_ASSERT_VALUES_EQUAL(*left.second, *right.second);
        }
    }

}
