#include "constructors.h"

#include <library/cpp/deprecated/atomic/atomic.h>
#include <util/string/cast.h>
#include <util/string/builder.h>
#include <util/digest/multi.h>

using namespace NYasm::NAldan;

namespace {
    static TAtomic NextVariableId = 0;

    ui64 GetNextVariableId() {
        ui64 oldValue = AtomicGet(NextVariableId);
        while (!AtomicGetAndCas(&NextVariableId, oldValue + 1, oldValue)) {
        }
        return oldValue;
    }

    TString TypesToString(const TVector<TTypeConstructor::TPtr>& types) {
        TStringBuilder builder;
        bool first = true;
        for (const auto& tp : types) {
            if (!first) {
                builder << ' ';
            }
            builder << tp->ToString();
            first = false;
        }
        return TString(builder);
    }
}

TTypeVariable::TTypeVariable()
    : Id(GetNextVariableId())
{
}

TString TTypeVariable::ToString() const {
    if (Instance) {
        return Instance->ToString();
    } else {
        return TString::Join('\'', ::ToString(Id));
    }
}

size_t TTypeVariable::Hash() const {
    return MultiHash(GetType(), Id);
}

TTypeConstructor::TPtr TTypeVariable::Prune() {
    if (Instance) {
        Instance = Instance->Prune();
        return Instance;
    } else {
        return this;
    }
}

bool TTypeVariable::EqualUnsafe(const TTypeConstructor& other) const {
    return Id == other.CastUnsafe<TTypeVariable>().Id;
}

TString TTypeOperator::ToString() const {
    if (Types.empty()) {
        return Name;
    } else {
        return TString::Join('(', Name, ' ', TypesToString(Types), ')');
    }
}

size_t TTypeOperator::Hash() const {
    size_t result = CombineHashes(THash<EType>()(GetType()), THash<TString>()(Name));
    for (const auto& tp : Types) {
        result = CombineHashes(result, THash<TTypeConstructor>()(*tp));
    }
    return result;
}

TTypeConstructor::TPtr TTypeOperator::Prune() {
    return this;
}

bool TTypeOperator::EqualUnsafe(const TTypeConstructor& other) const {
    const auto& typedOther(other.CastUnsafe<TTypeOperator>());
    if (Name != typedOther.Name || Types.size() != typedOther.Types.size()) {
        return false;
    }
    for (const auto index : xrange(Types.size())) {
        if (*Types[index] != *typedOther.Types[index]) {
            return false;
        }
    }
    return true;
}

TString TFunctionConstructor::ToString() const {
    return TString::Join('(', GetFromType()->ToString(), ' ', Name, ' ', GetToType()->ToString(), ')');
}
