#pragma once

#include <infra/yasm/aldan/common/inferred_types.h>

#include <util/generic/ptr.h>
#include <util/generic/string.h>
#include <util/generic/vector.h>
#include <util/generic/xrange.h>

namespace NYasm::NAldan {
    class TTypeVariable final : public TTypeConstructor {
    public:
        static const EType Type = EType::Variable;

        TTypeVariable();

        EType GetType() const override {
            return Type;
        }

        TString ToString() const override;

        size_t Hash() const override;

        TTypeConstructor::TPtr Prune() override;

        bool IsComposite() const override {
            return false;
        }

        TTypeConstructor::TPtr GetInstance() const {
            return Instance;
        }

        void SetInstance(TTypeConstructor::TPtr instance) {
            Instance.Reset(std::move(instance));
        }

    protected:
        bool EqualUnsafe(const TTypeConstructor& other) const override;

    private:
        const ui64 Id = 0;
        TTypeConstructor::TPtr Instance;
    };

    class TTypeOperator : public TTypeConstructor {
    public:
        static const EType Type = EType::Operator;

        TTypeOperator(const TString& name)
            : Name(name)
        {
        }

        TTypeOperator(const TString& name, TVector<TTypeConstructor::TPtr> types)
            : Name(name)
            , Types(std::move(types))
        {
        }

        virtual EType GetType() const override {
            return Type;
        }

        virtual TString ToString() const override;

        size_t Hash() const override;

        TTypeConstructor::TPtr Prune() override;

        bool IsComposite() const override {
            return true;
        }

        const TString& GetName() const {
            return Name;
        }

        const TVector<TTypeConstructor::TPtr>& GetTypes() const {
            return Types;
        }

    protected:
        bool EqualUnsafe(const TTypeConstructor& other) const override;

        const TString Name;
        TVector<TTypeConstructor::TPtr> Types;
    };

    class TFunctionConstructor final : public TTypeOperator {
    public:
        static const EType Type = EType::Function;

        TFunctionConstructor(TTypeConstructor::TPtr fromType, TTypeConstructor::TPtr toType)
            : TTypeOperator("->")
        {
            Types.reserve(2);
            Types.emplace_back(std::move(fromType));
            Types.emplace_back(std::move(toType));
        }

        EType GetType() const override {
            return Type;
        }

        TString ToString() const override;

        TTypeConstructor::TPtr GetFromType() const {
            return Types[0];
        }

        TTypeConstructor::TPtr GetToType() const {
            return Types[1];
        }
    };

    class TTupleConstructor final : public TTypeOperator {
    public:
        static const EType Type = EType::Tuple;

        TTupleConstructor(TVector<TTypeConstructor::TPtr> givenTypes)
            : TTypeOperator("tuple")
        {
            Types.swap(givenTypes);
        }

        EType GetType() const override {
            return Type;
        }
    };

    class TListConstructor final : public TTypeOperator {
    public:
        static const EType Type = EType::List;

        TListConstructor(TTypeConstructor::TPtr valueType)
            : TTypeOperator("list")
        {
            Types.reserve(1);
            Types.emplace_back(std::move(valueType));
        }

        EType GetType() const override {
            return Type;
        }

        TTypeConstructor::TPtr GetValueType() const {
            return Types[0];
        }
    };
}
