#include "parser.h"

#include <infra/yasm/aldan/common/exceptions.h>

#include <library/cpp/testing/unittest/registar.h>

using namespace NYasm::NAldan;

Y_UNIT_TEST_SUITE(TAldanParserTest) {

    TString Parse(const TString& expression) {
        const auto node = ParseExpression(expression);
        UNIT_ASSERT(node);
        return node->ToString();
    }

    Y_UNIT_TEST(TestGoodGrammar) {
        UNIT_ASSERT_VALUES_EQUAL(Parse("1.0"), "1");
        UNIT_ASSERT_VALUES_EQUAL(Parse("-1.0"), "-1");
        UNIT_ASSERT_VALUES_EQUAL(Parse("1.0 # comment should be ignored"), "1");
        UNIT_ASSERT_VALUES_EQUAL(Parse("signal_ahhh"), "signal_ahhh");
        UNIT_ASSERT_VALUES_EQUAL(Parse("const()"), "const()");
        UNIT_ASSERT_VALUES_EQUAL(Parse("const(max)"), "const(max)");
        UNIT_ASSERT_VALUES_EQUAL(Parse("1e10 + 1.1 + 2"), "plus(plus(1e+10, 1.1), 2)");
        UNIT_ASSERT_VALUES_EQUAL(Parse("1 + 2 - -3 * 4 / 5 ^ 6"), "minus(plus(1, 2), divide(multiply(-3, 4), power(5, 6)))");
        UNIT_ASSERT_VALUES_EQUAL(Parse("1 + ((2 - 3 * 4) / 5) ^ 6"), "plus(1, power(divide(minus(2, multiply(3, 4)), 5), 6))");
        UNIT_ASSERT_VALUES_EQUAL(Parse("1+((2-3*4)/5)^6"), "plus(1, power(divide(minus(2, multiply(3, 4)), 5), 6))");
        UNIT_ASSERT_VALUES_EQUAL(Parse("(((1.0 + 2.0)) + 3.0)"), "plus(plus(1, 2), 3)");
        UNIT_ASSERT_VALUES_EQUAL(Parse("func((1.0 + 2.0) * 3.0, const(), third(signal_axxx, 1))"),
                                 "func(multiply(plus(1, 2), 3), const(), third(signal_axxx, 1))");
        UNIT_ASSERT_VALUES_EQUAL(Parse(R"(
            # function comment
            const(
                max # argument comment
            )
        )"), "const(max)");
    }

    Y_UNIT_TEST(TestBadGrammar) {
        UNIT_ASSERT_EXCEPTION(Parse(""), TParsingError);
        UNIT_ASSERT_EXCEPTION(Parse(R"("const( max ))"), TParsingError);
        UNIT_ASSERT_EXCEPTION(Parse("const())"), TParsingError);
        UNIT_ASSERT_EXCEPTION(Parse("1 + +1"), TParsingError);
        UNIT_ASSERT_EXCEPTION(Parse("signal_<yasm|juggler>_ahhh"), TParsingError);
    }

    Y_UNIT_TEST(TestPatterns) {
        UNIT_ASSERT_VALUES_EQUAL(Parse("hsum(hmerge(signal_<yasm|juggler>_ahhh))"), "hsum(hmerge(signal_yasm_ahhh, signal_juggler_ahhh))");
    }
}
