#include "factory.h"
#include "instructions.h"

#include <infra/yasm/aldan/parser/parser.h>
#include <infra/yasm/aldan/infer/builtin_types.h>

#include <library/cpp/testing/unittest/registar.h>

using namespace NYasm::NAldan;
using namespace NZoom::NValue;

namespace {
    TTypeConstructor::TPtr CreateFunction(TTypeConstructor::TPtr argumentType, TTypeConstructor::TPtr returnType) {
        return MakeIntrusive<TFunctionConstructor>(argumentType, returnType);
    }

    TTypeConstructor::TPtr CreateTuple(std::initializer_list<TTypeConstructor::TPtr> arguments) {
        return MakeIntrusive<TTupleConstructor>(TVector<TTypeConstructor::TPtr>{arguments});
    }

    class TBinaryNumberDescriptor : public IFunctionDescriptor {
    public:
        TVector<TTypeConstructor::TPtr> GetFunctionTypes() const override {
            return TVector<TTypeConstructor::TPtr>{
                CreateFunction(CreateTuple({DOUBLE, INTEGER}), DOUBLE),
                CreateFunction(CreateTuple({INTEGER, DOUBLE}), DOUBLE),
                CreateFunction(CreateTuple({DOUBLE, DOUBLE}), DOUBLE)
            };
        }
    };

    class TDoublePlusDescriptor final : public TBinaryNumberDescriptor {
    public:
        TString GetFunctionName() const override {
            return "plus";
        }

        void GenerateInstructions(TVirtualMachine& virtualMachine) const override {
            virtualMachine.AddInstruction(TDoublePlusOp());
        }
    };

    class TDoubleMultiplyDescriptor final : public TBinaryNumberDescriptor {
    public:
        TString GetFunctionName() const override {
            return "multiply";
        }

        void GenerateInstructions(TVirtualMachine& virtualMachine) const override {
            virtualMachine.AddInstruction(TDoubleMultiplyOp());
        }
    };

    class TDoubleMinusDescriptor final : public TBinaryNumberDescriptor {
    public:
        TString GetFunctionName() const override {
            return "minus";
        }

        void GenerateInstructions(TVirtualMachine& virtualMachine) const override {
            virtualMachine.AddInstruction(TDoubleMinusOp());
        }
    };
}

Y_UNIT_TEST_SUITE(TAldanInstructionFactoryTest) {

    Y_UNIT_TEST(TestSimpleExpression) {
        auto rootNode = ParseExpression("42 - 8 * (7 + 5.0)");
        UNIT_ASSERT(rootNode);
        TInstructionFactory factory;
        factory.AddDescriptor(MakeHolder<TDoublePlusDescriptor>());
        factory.AddDescriptor(MakeHolder<TDoubleMinusDescriptor>());
        factory.AddDescriptor(MakeHolder<TDoubleMultiplyDescriptor>());
        TVirtualMachine machine;
        factory.GenerateInstructions(*rootNode, machine);
        UNIT_ASSERT_VALUES_EQUAL(machine.Run(), TValue(-54.0));
    }

}
