#pragma once

#include "value_stack.h"

#include <infra/yasm/common/points/accumulators/accumulators.h>
#include <infra/yasm/aldan/common/exceptions.h>

using namespace NYasm::NAldan;
using namespace NZoom::NValue;
using namespace NZoom::NHgram;

namespace {
    class TValueDisassemble final : public IUpdatable {
    public:
        TValueDisassemble(TVirtualMachineStack& stack)
            : Stack(stack)
        {
        }

        void MulNone() {
            Stack.Push();
        }

        void MulFloat(const double value) {
            Stack.Push(value);
        }

        void MulVec(const TVector<double>& value) {
            auto hgram = THgram::Default();
            hgram.MulSlice(value);
            Stack.Push(std::move(hgram));
        }

        void MulCountedSum(const double sum, const ui64 count) {
            Stack.Push(TSummary{
                .Sum = sum,
                .Count = count
            });
        }

        void MulHgram(const NZoom::NHgram::THgram& value) {
            auto hgram = THgram::Default();
            hgram.MulHgram(value);
            Stack.Push(std::move(hgram));
        }

    private:
        TVirtualMachineStack& Stack;
    };

    class TValueAssemble {
    public:
        static TValue Pack(TStackValue value) {
            return std::visit(TValueAssemble(), value);
        }

        TValue operator()(const std::monostate&) {
            return TValue();
        }

        TValue operator()(i64 value) {
            return TValue(value);
        }

        TValue operator()(double value) {
            return TValue(value);
        }

        TValue operator()(const TSummary& summary) {
            return TValue(summary.Sum, summary.Count);
        }

        TValue operator()(const THgram& hgram) {
            auto newHgram = THgram::Default();
            newHgram.MulHgram(hgram);
            return TValue(std::move(newHgram));
        }
    };

    class TIntegerReduction {
    public:
        using TValueType = i64;

        static TValueType Transform(TStackValue value) {
            return std::visit(TIntegerReduction(), value);
        }

        TValueType operator()(const std::monostate&) {
            return 0;
        }

        TValueType operator()(i64 value) {
            return value;
        }

        TValueType operator()(double value) {
            return value;
        }

        TValueType operator()(const TSummary& summary) {
            if (summary.Count) {
                return summary.Sum / summary.Count;
            } else {
                return 0;
            }
        }

        TValueType operator()(const THgram&) {
            ythrow TExecutionError() << "histogram can'be be converted to integer";
        }
    };

    class TDoubleReduction {
    public:
        using TValueType = double;

        static TValueType Transform(TStackValue value) {
            return std::visit(TDoubleReduction(), value);
        }

        TValueType operator()(const std::monostate&) {
            return 0.0;
        }

        TValueType operator()(i64 value) {
            return value;
        }

        TValueType operator()(double value) {
            return value;
        }

        TValueType operator()(const TSummary& summary) {
            if (summary.Count) {
                return summary.Sum / summary.Count;
            } else {
                return 0;
            }
        }

        TValueType operator()(const THgram&) {
            ythrow TExecutionError() << "histogram can'be be converted to double";
        }
    };
}

