#include "virtual_machine.h"
#include "operations.h"

using namespace NYasm::NAldan;

namespace {
    class TInstructionExecutor {
    public:
        TInstructionExecutor(TVirtualMachineStack& stack)
            : Stack(stack)
        {
        }

        void operator()(const TConstantIntegerOp& op) {
            Stack.Push(op.Value);
        }

        void operator()(const TConstantDoubleOp& op) {
            Stack.Push(op.Value);
        }

        void operator()(const TIntegerPlusOp&) {
            const auto args(BinaryCoerse<TIntegerReduction>());
            Stack.Push(args.first + args.second);
        }

        void operator()(const TDoublePlusOp&) {
            const auto args(BinaryCoerse<TDoubleReduction>());
            Stack.Push(args.first + args.second);
        }

        void operator()(const TIntegerMinusOp&) {
            const auto args(BinaryCoerse<TIntegerReduction>());
            Stack.Push(args.first - args.second);
        }

        void operator()(const TDoubleMinusOp&) {
            const auto args(BinaryCoerse<TDoubleReduction>());
            Stack.Push(args.first - args.second);
        }

        void operator()(const TIntegerDivideOp&) {
            const auto args(BinaryCoerse<TIntegerReduction>());
            Stack.Push(args.first / args.second);
        }

        void operator()(const TDoubleDivideOp&) {
            const auto args(BinaryCoerse<TDoubleReduction>());
            Stack.Push(args.first / args.second);
        }

        void operator()(const TIntegerMultiplyOp&) {
            const auto args(BinaryCoerse<TIntegerReduction>());
            Stack.Push(args.first * args.second);
        }

        void operator()(const TDoubleMultiplyOp&) {
            const auto args(BinaryCoerse<TDoubleReduction>());
            Stack.Push(args.first * args.second);
        }

    private:
        template <class TReduction>
        std::pair<typename TReduction::TValueType, typename TReduction::TValueType> BinaryCoerse() {
            auto right(TReduction::Transform(Stack.Pop()));
            auto left(TReduction::Transform(Stack.Pop()));
            return std::make_pair(std::move(left), std::move(right));
        }

        TVirtualMachineStack& Stack;
    };
}

TValue TVirtualMachine::Run() {
    TInstructionExecutor executor(Stack);
    for (const auto& instruction : Instructions) {
        std::visit(executor, instruction.Instruction);
    }
    if (Stack.Empty()) {
        ythrow TParsingError() << "stack is empty";
    }
    return TValueAssemble::Pack(Stack.Pop());
}
