# coding: utf-8

from libcpp cimport bool

from util.generic.string cimport TString, TStringBuf
from util.generic.vector cimport TVector


cdef extern from "infra/yasm/collector/python/deserializers/hserver.h" namespace "NCollector" nogil:
    cdef cppclass THserverResponse:
        THserverResponse(TStringBuf incoming) except +


cdef extern from "infra/yasm/collector/python/transmitters/clickhouse.h" namespace "NCollector" nogil:
    ctypedef struct TClickhouseOptions:
        TString Host
        int Port
        int Timeout
        TString DefaultDatabase
        TString User
        TString Password

        TString InstancesTableName
        TString SignalsTableName
        TString CommonTableName

        TVector[TString] Tags

        bool Dummy

    ctypedef struct TGroupInfo:
        TString GroupName
        TString MetaGroupName

    cdef cppclass TClickhouseTransmitter:
        TClickhouseTransmitter() except +
        TClickhouseTransmitter(const TClickhouseOptions& options) except +

        void HistoryExtend(const TVector[TGroupInfo]& groups, const THserverResponse& response) except +
        void RealtimeExtend(const TVector[TGroupInfo]& groups, const THserverResponse& response) except +

        void DumpInstances(bool force) except +
        void DumpSignals(bool force) except +
        void DumpCommon() except +

        void Reset() except +


cdef class HserverResponse:
    cdef THserverResponse* c_response

    def __cinit__(self, incoming):
        cdef TStringBuf buf = incoming
        with nogil:
            self.c_response = new THserverResponse(buf)

    def __dealloc__(self):
        del self.c_response


cdef class ClickhouseTransmitter:
    cdef TClickhouseTransmitter* c_transmitter

    def __cinit__(self, instances_table, signals_table, common_table,
                  host=None, port=None, user=None, password=None, database=None, timeout=None,
                  tags=None, dummy=None):
        cdef TClickhouseOptions c_options

        c_options.InstancesTableName = instances_table
        c_options.SignalsTableName = signals_table
        c_options.CommonTableName = common_table

        c_options.Host = host or "localhost"
        c_options.Port = port or 9000
        c_options.Timeout = timeout or 5 * 60
        c_options.DefaultDatabase = database or "default"
        c_options.User = user or "default"
        c_options.Password = password or ""

        c_options.Tags.clear()
        for tag in tags or ("ctype", "prj", "geo", "tier"):
            c_options.Tags.push_back(tag);

        c_options.Dummy = dummy or False

        self.c_transmitter = new TClickhouseTransmitter(c_options)

    def __dealloc__(self):
        del self.c_transmitter

    cdef fill_groups(self, groups, TVector[TGroupInfo]& c_groups):
        cdef TGroupInfo c_group_info
        for group, metagroup in groups:
            c_group_info.GroupName = group
            c_group_info.MetaGroupName = metagroup
            c_groups.push_back(c_group_info)

    def history_extend(self, groups, HserverResponse response):
        cdef TVector[TGroupInfo] c_groups
        self.fill_groups(groups, c_groups)

        cdef THserverResponse* c_response = response.c_response
        if c_response != NULL:
            with nogil:
                self.c_transmitter.HistoryExtend(c_groups, c_response[0])

    def realtime_extend(self, groups, HserverResponse response):
        cdef TVector[TGroupInfo] c_groups
        self.fill_groups(groups, c_groups)

        cdef THserverResponse* c_response = response.c_response
        if c_response != NULL:
            with nogil:
                self.c_transmitter.RealtimeExtend(c_groups, c_response[0])

    def dump_instances(self, force=False):
        cdef bool c_force = force
        with nogil:
            self.c_transmitter.DumpInstances(c_force)

    def dump_signals(self, force=False):
        cdef bool c_force = force
        with nogil:
            self.c_transmitter.DumpSignals(c_force)

    def dump_common(self):
        with nogil:
            self.c_transmitter.DumpCommon()

    def reset(self):
        with nogil:
            self.c_transmitter.Reset()
