#include "hserver.h"
#include "string_cache.h"

#include <infra/monitoring/common/msgpack.h>

#include <library/cpp/blockcodecs/codecs.h>

#include <util/generic/maybe.h>

#include <contrib/libs/msgpack/include/msgpack.hpp>

namespace NCollector {
    namespace {
        const static TStringBuf HIST_ROOT = "siglist_compact";
        const static TStringBuf RT_ROOT = "rt_siglist";
        const static TStringBuf CONTENT_ROOT = "content";

        const static TStringBuf HOSTS = "hosts";
        const static TStringBuf GROUPS = "groups";
        const static TStringBuf SIGNALS = "signals";

        const static NBlockCodecs::ICodec* SNAPPY_CODEC = NBlockCodecs::Codec("snappy");

        static TStringCache StringCache;

        TBuffer UnpackResponse(TStringBuf incoming) {
            TBuffer decompressed;
            SNAPPY_CODEC->Decode(incoming, decompressed);
            return decompressed;
        }

        using NMonitoring::TMapIterator;
        using NMonitoring::TArrayIterator;
        using NMonitoring::EnsureIs;

        TMaybe<msgpack::object_map> FindChild(msgpack::object_map response, TStringBuf expected) {
            for (const auto& item : TMapIterator(response)) {
                TStringBuf key(item.key.as<TStringBuf>());
                if (key == expected) {
                    EnsureIs(item.val, msgpack::type::MAP);
                    return item.val.via.map;
                }
            }
            return Nothing();
        }

        TMaybe<msgpack::object_map> FindChild(const msgpack::object& response, TStringBuf expected) {
            EnsureIs(response, msgpack::type::MAP);
            return FindChild(response.via.map, expected);
        }

        struct TResponseDeserializer {
            TResponseDeserializer(msgpack::object_map root)
            {
                HostSignalOffsets.reserve(100UL * 1000UL);

                for (const auto& section : TMapIterator(root)) {
                    TStringBuf key(section.key.as<TStringBuf>());
                    if (key == HOSTS) {
                        EnsureIs(section.val, msgpack::type::MAP);
                        ParseHosts(section.val.via.map);
                    } else if (key == GROUPS) {
                        EnsureIs(section.val, msgpack::type::MAP);
                        ParseGroups(section.val.via.map);
                    } else if (key == SIGNALS) {
                        EnsureIs(section.val, msgpack::type::ARRAY);
                        ParseSignals(section.val.via.array);
                    } else {
                        ythrow yexception() << "unknown key " << key << " given";
                    }
                }

                Finish();
            }

            void ParseHosts(msgpack::object_map hosts) {
                for (const auto& hostTags : TMapIterator(hosts)) {
                    TStringBuf hostName(StringCache.Get(hostTags.key.as<TStringBuf>()));
                    EnsureIs(hostTags.val, msgpack::type::MAP);
                    ParseTags(hostName, hostTags.val.via.map);
                }
            }

            void ParseTags(TStringBuf hostName, msgpack::object_map tags) {
                if (HostSignalOffsets.capacity() < HostSignalOffsets.size() + tags.size) {
                    HostSignalOffsets.reserve(HostSignalOffsets.size() + tags.size);
                }
                for (const auto& tagSignals : TMapIterator(tags)) {
                    EnsureIs(tagSignals.val, msgpack::type::ARRAY);

                    auto instanceKey(NTags::TInstanceKey::FromNamed(tagSignals.key.as<TStringBuf>()));
                    HostSignalOffsets.emplace_back(hostName, instanceKey, tagSignals.val.via.array);
                }
            }

            void ParseGroups(msgpack::object_map groups) {
                Groups.reserve(groups.size);
                for (const auto& groupHostNames : TMapIterator(groups)) {
                    EnsureIs(groupHostNames.val, msgpack::type::ARRAY);

                    TStringBuf group(StringCache.Get(groupHostNames.key.as<TStringBuf>()));
                    TVector<TStringBuf> hostNames;
                    hostNames.reserve(groupHostNames.val.via.array.size);
                    for (const auto& hostName : TArrayIterator(groupHostNames.val.via.array)) {
                        hostNames.emplace_back(StringCache.Get(hostName.as<TStringBuf>()));
                    }
                    Groups.emplace(group, std::move(hostNames));
                }
            }

            void ParseSignals(msgpack::object_array signals) {
                Signals.resize(signals.size);
                auto it(Signals.begin());
                for (const auto& signalName : TArrayIterator(signals)) {
                    *it = StringCache.Get(signalName.as<TStringBuf>());
                    ++it;
                }
            }

            void Finish() {
                HostSignals.reserve(HostSignalOffsets.size());
                for (const auto& item : HostSignalOffsets) {
                    HostSignals.emplace_back(std::get<0>(item), std::get<1>(item));

                    auto& result(HostSignals.back());
                    result.GetSignals().resize(std::get<2>(item).size);
                    auto it(result.GetSignals().begin());
                    for (const auto& offset : TArrayIterator(std::get<2>(item))) {
                        *it = Signals.at(offset.as<size_t>());
                        ++it;
                    }
                }
            }

            // hostname, instance key, signal offsets
            TVector<std::tuple<TStringBuf, NTags::TInstanceKey, msgpack::object_array>> HostSignalOffsets;
            // signal, use offset to get proper one
            TVector<TStringBuf> Signals;
            // group to hosts
            THashMap<TStringBuf, TVector<TStringBuf>> Groups;
            // result, will be moved into response
            TVector<THserverResponse::THostSignals> HostSignals;
        };
    }

    THserverResponse::THserverResponse(TStringBuf incoming)
    {
        auto buf(UnpackResponse(incoming));

        msgpack::zone zone(4UL * 1024UL * 1024UL);
        msgpack::object msg(msgpack::unpack(zone, buf.Data(), buf.Size()));

        auto root = FindChild(msg, RT_ROOT);
        if (root.Empty()) {
            root = FindChild(msg, HIST_ROOT);
        }
        if (root.Empty()) {
            ythrow yexception() << "can't find root in response";
        }

        root = FindChild(root.GetRef(), CONTENT_ROOT);
        if (root.Empty()) {
            ythrow yexception() << "can't find content in response";
        }

        TResponseDeserializer deserializer(root.GetRef());

        Groups = std::move(deserializer.Groups);
        HostSignals = std::move(deserializer.HostSignals);
    }
}
